<?php
/**
 * Internationalization: Stripe
 *
 * @package SimplePay\Core\i18n
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.9.0
 */

namespace SimplePay\Core\i18n;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Returns Stripe's supported countries.
 *
 * @since 3.9.0
 *
 * @return array
 */
function get_stripe_countries() {
	$countries = array(
		'AU' => __( 'Australia', 'simple-pay' ),
		'AT' => __( 'Austria', 'simple-pay' ),
		'BE' => __( 'Belgium', 'simple-pay' ),
		'BG' => __( 'Bulgaria', 'simple-pay' ),
		'CA' => __( 'Canada', 'simple-pay' ),
		'CY' => __( 'Cyprus', 'simple-pay' ),
		'CZ' => __( 'Czech Republic', 'simple-pay' ),
		'DK' => __( 'Denmark', 'simple-pay' ),
		'EE' => __( 'Estonia', 'simple-pay' ),
		'FI' => __( 'Finland', 'simple-pay' ),
		'FR' => __( 'France', 'simple-pay' ),
		'DE' => __( 'Germany', 'simple-pay' ),
		'GR' => __( 'Greece', 'simple-pay' ),
		'HK' => __( 'Hong Kong', 'simple-pay' ),
		'IE' => __( 'Ireland', 'simple-pay' ),
		'IT' => __( 'Italy', 'simple-pay' ),
		'JP' => __( 'Japan', 'simple-pay' ),
		'LV' => __( 'Latvia', 'simple-pay' ),
		'LT' => __( 'Lithuania', 'simple-pay' ),
		'LU' => __( 'Luxembourg', 'simple-pay' ),
		'MY' => __( 'Malaysia', 'simple-pay' ),
		'MT' => __( 'Malta', 'simple-pay' ),
		'MX' => __( 'Mexico', 'simple-pay' ),
		'NL' => __( 'Netherlands', 'simple-pay' ),
		'NZ' => __( 'New Zealand', 'simple-pay' ),
		'NO' => __( 'Norway', 'simple-pay' ),
		'PL' => __( 'Poland', 'simple-pay' ),
		'PT' => __( 'Portugal', 'simple-pay' ),
		'RO' => __( 'Romania', 'simple-pay' ),
		'SG' => __( 'Singapore', 'simple-pay' ),
		'SK' => __( 'Slovakia', 'simple-pay' ),
		'SI' => __( 'Slovenia', 'simple-pay' ),
		'ES' => __( 'Spain', 'simple-pay' ),
		'SE' => __( 'Sweden', 'simple-pay' ),
		'CH' => __( 'Switzerland', 'simple-pay' ),
		'GB' => __( 'United Kingdom', 'simple-pay' ),
		'US' => __( 'United States', 'simple-pay' ),
	);

	/**
	 * Filters the countries supported by Stripe.
	 *
	 * @since 3.9.0
	 *
	 * @param array $countries Country list, keyed by country code.
	 */
	$countries = apply_filters( 'simpay_get_stripe_countries', $countries );

	return $countries;
}

/**
 * Returns Stripe Checkout's supported locales.
 *
 * @since 3.9.0
 *
 * @return array
 */
function get_stripe_checkout_locales() {
	return array(
		'auto' => __( 'Auto-detect', 'simple-pay' ),
		'cs'   => __( 'Czech (cs)', 'simple-pay' ),
		'da'   => __( 'Danish (da)', 'simple-pay' ),
		'de'   => __( 'German (de)', 'simple-pay' ),
		'el'   => __( 'Greek (el)', 'simple-pay' ),
		'et'   => __( 'Estonian (et)', 'simple-pay' ),
		'en'   => __( 'English (en)', 'simple-pay' ),
		'es'   => __( 'Spanish (es)', 'simple-pay' ),
		'fi'   => __( 'Finnish (fi)', 'simple-pay' ),
		'fr'   => __( 'French (fr)', 'simple-pay' ),
		'hu'   => __( 'Hungarian (hu)', 'simple-pay' ),
		'it'   => __( 'Italian (it)', 'simple-pay' ),
		'ja'   => __( 'Japanese (ja)', 'simple-pay' ),
		'lt'   => __( 'Lithuanian (lt)', 'simple-pay' ),
		'lv'   => __( 'Latvian (lv)', 'simple-pay' ),
		'ms'   => __( 'Malay (ms)', 'simple-pay' ),
		'mt'   => __( 'Maltese (mt)', 'simple-pay' ),
		'nb'   => __( 'Norwegian Bokmål (nb)', 'simple-pay' ),
		'nl'   => __( 'Dutch (nl)', 'simple-pay' ),
		'pl'   => __( 'Polish (pl)', 'simple-pay' ),
		'pt'   => __( 'Portuguese (pt)', 'simple-pay' ),
		'ro'   => __( 'Romanian (ro)', 'simple-pay' ),
		'ru'   => __( 'Russian (ru)', 'simple-pay' ),
		'sk'   => __( 'Slovak (sk)', 'simple-pay' ),
		'sl'   => __( 'Slovenian (sl)', 'simple-pay' ),
		'sv'   => __( 'Swedish (sv)', 'simple-pay' ),
		'tk'   => __( 'Turkish (tk)', 'simple-pay' ),
		'zh'   => __( 'Chinese Simplified (zh)', 'simple-pay' ),
	);
}

/**
 * Returns Stripe Element's supported locales.
 *
 * @since 3.9.0
 *
 * @return array
 */
function get_stripe_elements_locales() {
	return array(
		'auto' => __( 'Auto-detect', 'simple-pay' ),
		'ar'   => __( 'Arabic (ar)', 'simple-pay' ),
		'bg'   => __( 'Bulgarian (bg)', 'simple-pay' ),
		'cs'   => __( 'Czech (cs)', 'simple-pay' ),
		'da'   => __( 'Danish (da)', 'simple-pay' ),
		'de'   => __( 'German (de)', 'simple-pay' ),
		'el'   => __( 'Greek (el)', 'simple-pay' ),
		'et'   => __( 'Estonian (et)', 'simple-pay' ),
		'en'   => __( 'English (en)', 'simple-pay' ),
		'es'   => __( 'Spanish (es)', 'simple-pay' ),
		'fi'   => __( 'Finnish (fi)', 'simple-pay' ),
		'fr'   => __( 'French (fr)', 'simple-pay' ),
		'he'   => __( 'Hebrew (he)', 'simple-pay' ),
		'in'   => __( 'Indonesian (in)', 'simple-pay' ),
		'it'   => __( 'Italian (it)', 'simple-pay' ),
		'ja'   => __( 'Japanese (ja)', 'simple-pay' ),
		'lt'   => __( 'Lithuanian (lt)', 'simple-pay' ),
		'lv'   => __( 'Latvian (lv)', 'simple-pay' ),
		'ms'   => __( 'Malay (ms)', 'simple-pay' ),
		'nb'   => __( 'Norwegian Bokmål (nb)', 'simple-pay' ),
		'nl'   => __( 'Dutch (nl)', 'simple-pay' ),
		'pl'   => __( 'Polish (pl)', 'simple-pay' ),
		'pt'   => __( 'Portuguese (pt)', 'simple-pay' ),
		'ru'   => __( 'Russian (ru)', 'simple-pay' ),
		'sk'   => __( 'Slovak (sk)', 'simple-pay' ),
		'sl'   => __( 'Slovenian (sl)', 'simple-pay' ),
		'sv'   => __( 'Swedish (sv)', 'simple-pay' ),
		'zh'   => __( 'Chinese Simplified (zh)', 'simple-pay' ),
	);
}

/**
 * Returns a list of error codes and corresponding localized error messages.
 *
 * @since 3.9.0
 *
 * @return array $error_list List of error codes and corresponding error messages.
 */
function get_localized_error_messages() {
	$error_list = array(
		'invalid_number'           => __( 'The card number is not a valid credit card number.', 'simple-pay' ),
		'invalid_expiry_month'     => __( 'The card\'s expiration month is invalid.', 'simple-pay' ),
		'invalid_expiry_year'      => __( 'The card\'s expiration year is invalid.', 'simple-pay' ),
		'invalid_cvc'              => __( 'The card\'s security code is invalid.', 'simple-pay' ),
		'incorrect_number'         => __( 'The card number is incorrect.', 'simple-pay' ),
		'incomplete_number'        => __( 'The card number is incomplete.', 'simple-pay' ),
		'incomplete_cvc'           => __( 'The card\'s security code is incomplete.', 'simple-pay' ),
		'incomplete_expiry'        => __( 'The card\'s expiration date is incomplete.', 'simple-pay' ),
		'expired_card'             => __( 'The card has expired.', 'simple-pay' ),
		'incorrect_cvc'            => __( 'The card\'s security code is incorrect.', 'simple-pay' ),
		'incorrect_zip'            => __( 'The card\'s zip code failed validation.', 'simple-pay' ),
		'invalid_expiry_year_past' => __( 'The card\'s expiration year is in the past', 'simple-pay' ),
		'card_declined'            => __( 'The card was declined.', 'simple-pay' ),
		'processing_error'         => __( 'An error occurred while processing the card.', 'simple-pay' ),
		'invalid_request_error'    => __( 'Unable to process this payment, please try again or use alternative method.', 'simple-pay' ),
		'email_invalid'            => __( 'Invalid email address, please correct and try again.', 'simple-pay' ),
	);

	/**
	 * Filters the list of available error codes and corresponding error messages.
	 *
	 * @since 3.9.0
	 *
	 * @param array $error_list List of error codes and corresponding error messages.
	 */
	$error_list = apply_filters( 'simpay_get_localized_error_list', $error_list );

	return $error_list;
}

/**
 * Returns a localized error message for a corresponding Stripe
 * error code.
 *
 * @link https://stripe.com/docs/error-codes
 *
 * @since 3.9.0
 *
 * @param string $error_code Error code.
 * @param string $error_message Original error message to return if a localized version does not exist.
 * @return string $error_message Potentially localized error message.
 */
function get_localized_error_message( $error_code, $error_message ) {
	$error_list = get_localized_error_messages();

	if ( isset( $error_list[ $error_code ] ) ) {
		return $error_list[ $error_code ];
	}

	return $error_message;
}
