<?php
/**
 * REST API: Checkout Session Controller
 *
 * @package SimplePay\Core\REST_API\v2
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Core\REST_API\v2;

use SimplePay\Core\Payments;
use SimplePay\Core\Forms\Default_Form;
use SimplePay\Core\REST_API\Controller;
use SimplePay\Core\Legacy;
use SimplePay\Core\Utils;


// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Checkout_Session_Controller.
 */
class Checkout_Session_Controller extends Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpsp/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'checkout-session';

	/**
	 * Registers the routes for Checkout Session.
	 *
	 * @since 3.6.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Allows POST requests to this endpoint with a valid nonce.
	 *
	 * @since 3.6.0
	 *
	 * @param \WP_REST_Request Request data.
	 * @return bool True with a valid nonce.
	 */
	public function create_item_permissions_check( $request ) {
		$form_values = $request['form_values'];

		if ( ! isset( $form_values['_wpnonce'] ) || ! wp_verify_nonce( $form_values['_wpnonce'], 'simpay_payment_form' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Handles an incoming request to create a Checkout\Session.
	 *
	 * @since 3.6.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type int   $customer_id Customer ID previously generated with Payment Source.
	 *   @type int   $form_id Form ID used to generate PaymentIntent data.
	 *   @type array $form_data Client-generated formData information.
	 *   @type array $form_values Values of named fields in the payment form.
	 * }
	 * @return \WP_REST_Response
	 */
	public function create_item( $request ) {
		try {
			// Gather customer information.
			$customer_id = isset( $request['customer_id'] ) ? $request['customer_id'] : '';

			// Locate form.
			if ( ! isset( $request['form_id'] ) ) {
				throw new \Exception( __( 'Unable to locate payment form.', 'simple-pay' ) );
			}

			// Gather <form> information.
			$form_id     = $request['form_id'];
			$form_data   = $request['form_data'];
			$form_values = $request['form_values'];

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			// Handle legacy form processing.
			Legacy\Hooks\simpay_process_form( $form, $form_data, $form_values, $customer_id );

			$session_args = $this->get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id );

			/**
			 * Allows processing before a Checkout\Session is created from a payment form request.
			 *
			 * @since 3.6.0
			 *
			 * @param array                         $session_args Arguments used to create a PaymentIntent.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int|string                    $customer_id Stripe Customer ID, or a blank string if none is needed.
			 */
			do_action(
				'simpay_before_checkout_session_from_payment_form_request',
				$session_args,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			$session = Payments\Stripe_Checkout\Session\create(
				$session_args,
				$form->get_api_request_args()
			);

			/**
			 * Allows further processing after a Checkout\Session is created from a payment form request.
			 *
			 * @since 3.6.0
			 *
			 * @param \Stripe\Checkout\Sesssion     $session Stripe Checkout Session.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int                           $customer_id Stripe Customer ID.
			 */
			do_action(
				'simpay_after_checkout_session_from_payment_form_request',
				$session,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			return new \WP_REST_Response(
				array(
					'sessionId' => $session->id,
					'session'   => $session,
				)
			);
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => Utils\handle_exception_message( $e ),
				),
				400
			);
		}
	}

	/**
	 * Uses the current payment form request to generate arguments for a Checkout Session.
	 *
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	function get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
		$session_args = array(
			'locale'               => $form->locale,
			'metadata'             => array(
				'simpay_form_id' => $form->id,
			),
			'payment_method_types' => array(
				'card',
			),
		);

		// Collect Billing Address.
		if ( true === $form->enable_billing_address ) {
			$session_args['billing_address_collection'] = 'required';
		} else {
			$session_args['billing_address_collection'] = 'auto';
		}

		// Collect Shipping Address.
		if ( true === $form->enable_shipping_address ) {
			$session_args['shipping_address_collection'] = array(
				'allowed_countries' => Payments\Stripe_Checkout\get_available_shipping_address_countries(),
			);
		}

		// Attach a Customer.
		if ( ! empty( $customer_id ) ) {
			$session_args['customer'] = $customer_id;
		}

		// Success URL

		// Escape base URL.
		$session_args['success_url'] = esc_url_raw( $form->payment_success_page );

		// Ensure a valid base URL exists.
		if ( empty( $session_args['success_url'] ) ) {
			$session_args['success_url'] = esc_url_raw( home_url() );
		}

		// Avoid escaping the {CHECKOUT_SESSION_ID} tag.
		$session_args['success_url'] = add_query_arg( 'session_id', '{CHECKOUT_SESSION_ID}', $session_args['success_url'] );

		// Cancel URL

		// Escape base URL.
		$session_args['cancel_url'] = esc_url_raw( $form->payment_cancelled_page );

		// Ensure a valid base URL exists.
		if ( empty( $session_args['cancel_url'] ) || false === $session_args['cancel_url'] ) {
			$session_args['cancel_url'] = esc_url_raw( home_url() );
		}

		// Submit type.
		if ( isset( $form->checkout_submit_type ) ) {
			$session_args['submit_type'] = $form->checkout_submit_type;
		}

		// Add a line item.
		$amount = isset( $form_values['simpay_amount'] )
			? $form_values['simpay_amount']
			: simpay_convert_amount_to_cents( $form->amount );

		$images = null;

		if ( '' !== $form->image_url ) {
			$images = array(
				esc_url( $form->image_url ),
			);
		}

		$quantity = isset( $form_values['simpay_quantity'] )
			? absint( $form_values['simpay_quantity'] )
			: 1;

		$name = ! empty( $form->company_name )
			? $form->company_name
			: get_bloginfo( 'name' );

		$name = ! empty( $name )
			? $name
			: 'WP Simple Pay';

		$line_item = array(
			'amount'      => $amount / $quantity,
			'currency'    => $form->currency,
			'name'        => $name,
			'quantity'    => $quantity,
			'description' => Legacy\Hooks\simpay_payment_description( $form, $form_data, $form_values, $customer_id ),
			'images'      => $images,
		);

		$session_args['line_items'] = array(
			$line_item,
		);

		// Add PaymentIntent data.
		$payment_intent_data = Payments\PaymentIntent\get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id );

		// Remove unsupported parameters for Checkout.
		unset( $payment_intent_data['currency'] );
		unset( $payment_intent_data['amount'] );

		$session_args['payment_intent_data'] = $payment_intent_data;

		/**
		 * Filters arguments used to create a Checkout Session from a payment form request.
		 *
		 * @since 3.6.0
		 *
		 * @param array                         $session_args Checkout Session args.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		return apply_filters(
			'simpay_get_session_args_from_payment_form_request',
			$session_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);
	}
}
