/* global _ */

/**
 * WordPress dependencies
 */
import domReady from '@wordpress/dom-ready';

/**
 * Internal dependencies
 */
const { hooks } = window.wpsp;

/**
 * Update field list when necessary.
 */
domReady( () => {
	updateCustomFieldList();

	// Update list when settings change.
	_.each( [
		...document.querySelectorAll( 'input[name="_form_display_type"]' ),
		...document.querySelectorAll( 'input[name="_amount_type"]' ),
		...document.querySelectorAll( 'input[name="_subscription_type"]' ),
	], ( input ) => {
		input.addEventListener( 'change', updateCustomFieldList );
	} );
} );

hooks.addAction( 'customFieldAdded', 'wpsp/payment-form', updateCustomFieldList );
hooks.addAction( 'customFieldRemoved', 'wpsp/payment-form', updateCustomFieldList );

/**
 * Updates the field list to set certain fields disabled or enabled.
 */
function updateCustomFieldList() {
	const customFieldSelector = document.getElementById( 'custom-field-select' );

	if ( ! customFieldSelector ) {
		return;
	}

	const options = customFieldSelector.querySelectorAll( 'option' );

	_.each( options, ( option ) => {
		const disabled = hooks.applyFilters( 'isCustomFieldDisabled', false, option );
		option.disabled = disabled;
	} );
}

/**
 * Determines if a one-time use field is disabled.
 *
 * @param {boolean} disabled If the option to add the field is disabled.
 */
hooks.addFilter( 'isCustomFieldDisabled', 'wpsp/payment-form', ( disabled, option ) => {
	const existingField = document.querySelector( `.simpay-custom-field-${ option.value.replace( /_/g, '-' ) }` );
	const repeatable = 'true' === option.dataset.repeatable;

	if ( repeatable ) {
		return disabled;
	}

	return null !== existingField;
}, 10 );

/**
 * Determines if the "Payment Button" is disabled.
 *
 * @param {boolean} disabled If the option to add the field is disabled.
 */
hooks.addFilter( 'isCustomFieldDisabled', 'wpsp/payment-form', ( disabled, option ) => {
	if ( 'payment_button' !== option.value ) {
		return disabled;
	}

	if ( true === disabled ) {
		return disabled;
	}

	const formDisplayType = document.querySelector( 'input[name="_form_display_type"]:checked' ).value;

	return 'overlay' !== formDisplayType;
}, 20 );

/**
 * Determines if Subscription-specific fields should be disabled.
 *
 * @param {boolean} disabled If the option to add the field is disabled.
 */
hooks.addFilter( 'isCustomFieldDisabled', 'wpsp/payment-form', ( disabled, option ) => {
	const fieldType = option.value;
	const subscriptionFields = [
		'plan_select',
		'custom_amount',
		'recurring_amount_toggle',
	];

	if ( ! subscriptionFields.includes( fieldType ) ) {
		return disabled;
	}

	const subscriptionType = document.querySelector( 'input[name="_subscription_type"]:checked' ).value;

	if ( 'disabled' === subscriptionType ) {
		if ( 'plan_select' === fieldType ) {
			return true;
		}
	} else {
		if ( 'recurring_amount_toggle' === fieldType || 'custom_amount' === fieldType ) {
			return true;
		}
	}

	return disabled;
}, 30 );

/**
 * Determines if "Custom Amount" is disabled.
 *
 * @param {boolean} disabled If the option to add the field is disabled.
 */
hooks.addFilter( 'isCustomFieldDisabled', 'wpsp/payment-form', ( disabled, option ) => {
	if ( 'custom_amount' !== option.value ) {
		return disabled;
	}

	if ( true === disabled ) {
		return disabled;
	}

	const amountType = document.querySelector( 'input[name="_amount_type"]:checked' ).value;
	const subscriptionType = document.querySelector( 'input[name="_subscription_type"]:checked' ).value;

	return ! ( 'one_time_custom' === amountType && 'disabled' === subscriptionType );
}, 40 );

/**
 * Determines if the "Payment Methods" and "Checkout Button" is disabled
 * for Stripe Checkout.
 *
 * @param {boolean} disabled If the option to add the field is disabled.
 */
hooks.addFilter( 'isCustomFieldDisabled', 'wpsp/payment-form', ( disabled, option ) => {
	if ( ! [ 'card', 'checkout_button' ].includes( option.value ) ) {
		return disabled;
	}

	if ( true === disabled ) {
		return disabled;
	}

	const formDisplayType = document.querySelector( 'input[name="_form_display_type"]:checked' ).value;

	return 'stripe_checkout' === formDisplayType;
}, 50 );
