/* global jQuery, _ */

/**
 * Launch dialogs when "Configure" is clicked.
 *
 * @since 3.8.0
 *
 * @param {jQuery} $ jQuery
 */
( function( $ ) {
	$( document ).ready( () => {
		configure();
		restrict();
	} );

	function restrict() {
		const subscriptionTypes = document.querySelectorAll(
			'input[name="_subscription_type"]'
		);

		if ( 0 === subscriptionTypes.length ) {
			return;
		}

		_.each( subscriptionTypes, ( subscriptionType ) => {
			subscriptionType.addEventListener( 'change', onRestrict );
		} );

		/**
		 * Restricts the ability to enable Payment Methods based
		 * on the Subscription status.
		 *
		 * @since 3.8.0
		 *
		 * @param {Object} e Change event.
		 */
		function onRestrict( e ) {
			e.preventDefault();

			const hasSubscription = 'disabled' !== e.target.value;
			const paymentMethods = document.querySelectorAll(
				'.simpay-payment-method'
			);

			_.each( paymentMethods, ( paymentMethodInput ) => {
				// Do nothing if it supports Subscriptions.
				if ( undefined !== paymentMethodInput.dataset.recurring ) {
					return;
				}

				// Disable and uncheck if a Subscription is being used.
				if ( true === hasSubscription ) {
					paymentMethodInput.disabled = true;

					// Reenable if a Subscription is not being used.
				} else {
					// Only enable if not previously force-disabled.
					if ( undefined === paymentMethodInput.dataset.disabled ) {
						paymentMethodInput.disabled = false;
					}
				}
			} );
		}
	}

	/**
	 * Handles configuring a Payment Method in the "Payment Methods" custom form field.
	 *
	 * @since 3.8.0
	 */
	function configure() {
		const configureButtons = document.querySelectorAll(
			'.payment-method-configure'
		);

		if ( 0 === configureButtons.length ) {
			return;
		}

		_.each( configureButtons, ( button ) => {
			button.addEventListener( 'click', onConfigure );
		} );

		/**
		 * Opens a jQuery UI dialog to configure the Payment Method.
		 *
		 * @since 3.8.0
		 *
		 * @param {Event} e Click event.
		 */
		function onConfigure( e ) {
			e.preventDefault();

			const paymentMethod = e.target.dataset.paymentMethod;

			$( `#simpay-payment-method-configure-${ paymentMethod }` ).dialog( {
				position: {
					my: 'center',
					at: 'center',
				},
				width: '350px',
				modal: true,
				resizable: false,
				draggable: false,
				appendTo: e.target.parentElement,
				open( event ) {
					$( event.target )
						.find( '.update' )
						.on( 'click', ( e ) => {
							e.preventDefault();

							$( this ).dialog( 'close' );
						} );
				},
			} );
		}
	}
} )( jQuery );
