/* global simpayUpdatePaymentMethod, Stripe, jQuery, _ */

/**
 * External dependencies.
 */
import serialize from 'form-serialize';

/**
 * WordPress dependencies
 */
import domReady from '@wordpress/dom-ready';

/**
 * Internal dependencies.
 */
import { apiRequest } from '@wpsimplepay/utils';
import { onPaymentFormError } from '@wpsimplepay/core/frontend/payment-forms';
import { updatePaymentMethod } from '@wpsimplepay/pro/frontend/payments/subscription.js';
import { getPaymentMethod } from '@wpsimplepay/pro/frontend/payment-methods';
import { getPaymentForms } from '@wpsimplepay/pro/frontend/payment-forms';

const { disableForm } = window.simpayApp;

/**
 * Updates a Subscription with a new Source.
 *
 * @param {jQuery} spFormElem Form element jQuery object.
 * @param {Object} formData Configured form data.
 */
async function handleUpdatePaymentMethod( sourceId, spFormElem, formData ) {
	try {
		const { formId } = formData;
		const customerId = spFormElem.find( 'input[name="customer_id"]' ).val();
		const subscriptionId = spFormElem.find( 'input[name="subscription_id"]' ).val();
		const subscriptionKey = spFormElem.find( 'input[name="subscription_key"]' ).val();
		const _wpnonce = spFormElem.find( 'input[name="_wpnonce"]' ).val();

		// Update a Subscription's PaymentMethod
		await updatePaymentMethod(
			customerId,
			subscriptionId,
			formId,
			{
				source_id: sourceId,
				subscription_key: subscriptionKey,
				_wpnonce,
			}
		);

		window.location.replace( window.location.href );
	} catch ( error ) {
		onPaymentFormError( error, spFormElem, formData )
	}
}

// Custom "Update Payment Method" handlers for Payment Methods.
const createSource = {
	/**
	 * Handles Card updates.
	 *
	 * @param {jQuery} spFormElem Form element jQuery object.
	 * @param {Object} formData Configured form data.
	 */
	card: async ( spFormElem, formData ) => {
		// Access Stripe and Elements instance.
		const { stripeInstance: stripe, stripeInstance: { elements } } = spFormElem;

		try {
			const { error, source } = await stripe.createSource( elements.card, {
				type: 'card',
			} );

			// Bail if Source cannot be created.
			if ( error ) {
				throw error;
			}

			return source.id;
		} catch ( error ) {
			return onPaymentFormError( error, spFormElem, formData );
		}
	},
	/**
	 * Handles ACH Debit updates.
	 *
	 * @param {jQuery} spFormElem Form element jQuery object.
	 * @param {Object} formData Configured form data.
	 */
	'ach-debit': async ( spFormElem, formData ) => {
		const achDebitEl = spFormElem[ 0 ].querySelector( '.simpay-ach-debit-wrap' );
		const { publicToken, accountId } = achDebitEl.dataset;

		try {
			const { stripe_bank_account_token } = await apiRequest( 'v2/plaid-token', {
				form_values: serialize( spFormElem[ 0 ], { hash: true } ),
				form_data: formData,
				form_id: formData.formId,
				public_token: publicToken,
				account_id: accountId,
			} );

			return stripe_bank_account_token;
		} catch ( error ) {
			return onPaymentFormError( error, spFormElem, formData );
		}
	}
}

/**
 * Setup "Update Payment Method" form.
 */
function setupForm() {
	const formEl = document.getElementById( 'simpay-form-update-payment-method' );

	if ( ! formEl ) {
		return;
	}

	const {
		id,
		i18n,
	} = simpayUpdatePaymentMethod;

	// Deal with legacy script variables...
	const {
		[ id ]: {
			form: {
				config,
				bools,
				strings,
				i18n: formI18n,
			},
			stripe,
		}
	} = simplePayForms;

	// Mock spFormElem and formData.
	const formData = {
		formDisplayType: 'embedded',
		checkoutButtonLoadingText: i18n.loading,
		checkoutButtonText: i18n.submit,
		stripeParams: {
			key: stripe.strings.key,
		},
		formId: id,
		isTestMode: bools.isTestMode,
		companyName: strings.companyName,
		stripeErrorMessages: formI18n.stripeErrorMessages,
	};

	const spFormElem = jQuery( formEl );
	spFormElem.stripeInstance = Stripe( formData.stripeParams.key );

	// Mock a cart.
	spFormElem.cart = {};

	// Handle Payment Method change.
	// @todo Setup elsewhere.
	const { paymentMethods } = config;
	spFormElem.paymentMethod = paymentMethods[0].id;

	_.each( formEl.querySelectorAll( '.simpay-payment-method-toggle' ), ( paymentMethodToggle ) => {
		paymentMethodToggle.addEventListener( 'click', ( e ) => {
			spFormElem.paymentMethod = e.target.dataset.paymentMethod;
		} );
	} );

	// Setup the form's Payment Methods.
	paymentMethods.forEach( ( { id } ) => {
		getPaymentMethod( id ).config = _.findWhere( paymentMethods, {
			id,
		} );
		getPaymentMethod( id ).setup( spFormElem, formData );
	} );

	// Attach submission handler.
	formEl.addEventListener( 'submit', async ( e ) => {
		e.preventDefault();

		// Disable form while processing.
		disableForm( spFormElem, formData, true );

		try {
			const source = await createSource[ spFormElem.paymentMethod ]( spFormElem, formData );

			if ( undefined === source ) {
				return;
			}

			handleUpdatePaymentMethod( source, spFormElem, formData )
				.catch( ( error ) => {
					onPaymentFormError( error, spFormElem, formData );
				} );
		} catch ( error ) {
			onPaymentFormError( error, spFormElem, formData )
		}
	} );
}

// DOM Ready.
domReady( setupForm );
