<?php
/**
 * Forms field: Card
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.0.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Pro\Payment_Methods;
use SimplePay\Core\Abstracts\Custom_Field;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Card class.
 *
 * @since 3.0.0
 */
class Card extends Custom_Field {

	/**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.0.0
	 * @since 3.8.0 Passes field type as second arugment.
	 * @since 3.8.0 Passes form instance as third argument.
	 *
	 * @param array                         $settings Field settings.
	 * @param string                        $type Field type.
	 * @param SimplePay\Core\Abstracts\Form $form Form.
	 * @return string
	 */
	public static function print_html( $settings, $type, $form ) {
		$label = isset( $settings['label'] ) ? $settings['label'] : '';

		$payment_form    = $form;
		$payment_methods = Payment_Methods\get_form_payment_methods( $form );

		/**
		 * Allows output or action to be taken before ouputting the Payment Form's
		 * Payment Methods.
		 *
		 * @since 3.9.0
		 *
		 * @param \SimplePay\Pro\Payment_Methods\Payment_Method[] $payment_methods Payment Methods.
		 * @param \SimplePay\Core\Abstracts\Form                  $payment_form    Payment Form.
		 */
		do_action(
			'simpay_payment_form_payment_methods_before',
			$payment_methods,
			$payment_form
		);

		$payment_methods = array_values( $payment_methods );

		ob_start();
		?>

<div class="simpay-form-control simpay-form-control--card simpay-card-container">
		<?php echo self::get_label(); // WPCS: XSS okay. ?>

	<div class="simpay-form-tabs">
		<div
			class="simpay-form-tabs-toggles"
			role="tablist"
			aria-label="<?php echo esc_html( $label ); ?>"
			<?php if ( 1 === count( $payment_methods ) ) : ?>
				style="display: none;"
			<?php endif; ?>
		>
			<?php foreach ( $payment_methods as $i => $payment_method ) : ?>
				<button
					role="tab"
					aria-controls="simpay-payment-method-panel-<?php echo esc_attr( $payment_method->id ); ?>"
					id="simpay-payment-method-toggle-<?php echo esc_attr( $payment_method->id ); ?>"
					class="simpay-payment-method-toggle simpay-form-tabs-toggles__toggle <?php echo esc_attr( 0 === $i ? 'is-active' : '' ); ?>"
					tabindex="0"
					data-payment-method="<?php echo esc_attr( $payment_method->id ); ?>"
					<?php if ( 0 === $i ) : ?>
						aria-selected="true"
					<?php endif; ?>
				>
					<?php echo esc_html( $payment_method->nicename ); ?>
				</button>
			<?php endforeach; ?>
		</div>

		<?php foreach ( $payment_methods as $i => $payment_method ) : ?>
			<div
				id="simpay-payment-method-panel-<?php echo esc_attr( $payment_method->id ); ?>"
				role="tabpanel"
				tabindex="0"
				aria-labelledby="simpay-payment-method-toggle-<?php echo esc_attr( $payment_method->id ); ?>"
				<?php if ( $i > 0 ) : ?>
					hidden
				<?php endif; ?>
			>

				<?php
				switch ( $payment_method->id ) :
					case 'card':
						self::get_card_field();
						break;
					case 'ideal':
						self::get_ideal_field();
						break;
					case 'ach-debit':
						self::get_ach_debit_field();
						break;
				endswitch;
				?>

			</div>
		<?php endforeach; ?>
	</div>
</div>

		<?php
		return ob_get_clean();
	}

	/**
	 * Outputs markup for Card Payment Method.
	 *
	 * @since 3.9.0
	 */
	public static function get_card_field() {
		$id          = self::get_id_attr();
		$postal_code = isset( self::$settings['postal_code'] )
			? 'no'
			: 'yes';
		?>

		<div
			id="simpay-card-element-<?php echo esc_attr( $id ); ?>"
			class="simpay-card-wrap simpay-field-wrap"
			data-show-postal-code="<?php echo esc_attr( $postal_code ); ?>"
		>
		</div>

		<?php
	}

	/**
	 * Outputs markup for iDEAL Payment Method.
	 *
	 * @since 3.9.0
	 */
	public static function get_ideal_field() {
		$id = self::get_id_attr();
		?>

<div
	id="simpay-ideal-element-<?php echo esc_attr( $id ); ?>"
	class="simpay-ideal-wrap simpay-field-wrap"
>
</div>

		<?php
	}

	/**
	 * Outputs markup for ACH Debit Payment Method.
	 *
	 * @since 3.9.0
	 */
	public static function get_ach_debit_field() {
		$custom_fields = simpay_get_saved_meta( self::$form->id, '_custom_fields' );

		// Determine what to prepend "Select Bank" with.
		// Uses the "Checkout Button" custom field settings.
		$default_button_text = __( 'Pay {{amount}}', 'simple-pay' );
		$checkout_button     = isset( $custom_fields['checkout_button'] )
			? current( $custom_fields['checkout_button'] )
			: array(
				'text' => $default_button_text,
			);

		$checkout_button_text = empty( $checkout_button['text'] )
			? $default_button_text
			: $checkout_button['text'];
		$checkout_button_text = str_replace(
			'{{amount}}',
			(
				'<span>' . 
				simpay_format_currency(
					simpay_get_form_setting( 'amount', self::$form->id ),
					simpay_get_setting( 'currency' )
				) .
				'</span>'
			),
			$checkout_button_text
		);

		$id                 = self::get_id_attr();
		$ach_debit_settings = Payment_Methods\get_form_payment_method_settings( self::$form, 'ach-debit' );
		$plaid_mode         = false === self::$form->is_livemode()
			? 'sandbox'
			: 'production';

		$no_api_key_message = __( 'Invalid "Client ID" or "Secret" configured in Payment Methods settings for ACH payments.', 'simple-pay' );

		// If misconfigured and the current user can fix it, show an additional link.
		if ( current_user_can( 'manage_options' ) ) {
			$custom_fields_url = add_query_arg(
				array(
					'post'   => self::$form->id,
					'action' => 'edit',
				),
				admin_url( 'post.php#custom-form-fields-settings-panel' )
			);

			$no_api_key_message .= ' ' . (
				'<a href="' . esc_url( $custom_fields_url ) . '">' .
				esc_html__( 'Configure Payment Method', 'simple-pay' ) .
				'</a>'
			);
		}

		?>

<div
	id="simpay-ach-debit-<?php echo esc_attr( $id ); ?>"
	class="simpay-ach-debit-wrap simpay-field-wrap"
>
	<?php
	if ( empty( $ach_debit_settings['client_id'] ) || empty( $ach_debit_settings['secret_key'][ $plaid_mode ] ) ) :
		echo wp_kses(
			$no_api_key_message,
			array(
				'a' => array(
					'href' => true,
				)
			)
		);
	else :
	?>
	<button type="button" class="simpay-btn">
		<?php
		echo wp_kses(
			sprintf(
				/* translators: %s "Checkout Button" text prepended. */
				__( '%s - Select Bank', 'simple-pay' ),
				$checkout_button_text
			),
			array(
				'span' => true,
			)
		);
		?>
	</button>
	<?php endif; ?>
</div>

		<?php
	}

}
