<?php
/**
 * Forms field: Dropdown
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.0.0
 */

namespace SimplePay\Pro\Forms\Fields;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Dropdown class.
 *
 * @since 3.0.0
 */
class Dropdown extends Quantity_Amount {

	/**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		// Set shared properties.
		// Called here due to the legacy architecture of these fields.
		self::set_properties( $settings );

		$is_amount   = isset( $settings['amount_quantity'] ) && 'amount' === $settings['amount_quantity'];
		$is_quantity = isset( $settings['amount_quantity'] ) && 'quantity' === $settings['amount_quantity'];

		ob_start();
		?>

		<div class="simpay-form-control simpay-dropdown-container">
			<div class="simpay-dropdown-label simpay-label-wrap">
				<?php echo self::get_label(); // WPCS: XSS okay. ?>
				<div class="simpay-dropdown-wrap simpay-field-wrap">
					<?php
					if ( $is_amount ) :
						echo self::get_amount_dropdown( $settings );
					elseif ( $is_quantity ) :
						echo self::get_quantity_dropdown( $settings );
					else :
						echo self::get_standard_dropdown( $settings );
					endif;
					?>
				</div>
			</div>
		</div>

		<?php
		return ob_get_clean();
	}

	/**
	 * Returns HTML options for the dropdown.
	 *
	 * @since 3.9.0
	 */
	private static function get_standard_dropdown( $settings ) {
		$options        = self::get_options( $settings );
		$options_markup = '';

		foreach ( $options as $v ) {
			$options_markup .= '<option ' . selected( $v, self::$default, false ) . '>' . esc_html( $v ) . '</option>';
		}

		return (
			'<select name="' . esc_attr( self::$name ) . '" id="' . esc_attr( self::$id ) . '">' .
				$options_markup .
			'</select>'
		);
	}

	/**
	 * Returns HTML options for the Quantity dropdown.
	 *
	 * @since 3.9.0
	 *
	 * @return string
	 */
	private static function get_quantity_dropdown( $settings ) {
		$options    = self::get_options( $settings );
		$quantities = self::get_quantities( $settings );

		// Make sure the number of options and amounts is equal before continuing
		if ( count( $options ) !== count( $quantities ) ) {
			return (
				'<div style="color: red;">' .
					esc_html__( 'You have a mismatched number of options and amounts. Please correct this for the dropdown to appear.', 'simple-pay' ) .
				'</div>'
			);
		}

		$options_markup = '';

		foreach ( $options as $i => $v ) {
			$options_markup .= '<option ' . selected( $v, self::$default, false ) . ' data-quantity="' . esc_attr( intval( $quantities[ $i ] ) ) . '">' . esc_html( $v ) . '</option>';
		}

		return (
			'<select name="' . esc_attr( self::$name ) . '" id="' . esc_attr( self::$id ) . '" class="simpay-quantity-dropdown">' .
				$options_markup .
			'</select>' .
			'<input type="hidden" name="simpay_quantity" class="simpay-quantity" value="" />'
		);
	}

	/**
	 * Returns HTML options for the Amount dropdown.
	 *
	 * @since 3.9.0
	 *
	 * @return string
	 */
	private static function get_amount_dropdown( $settings ) {
		$options = self::get_options( $settings );
		$amounts = self::get_amounts( $settings );

		// Make sure the number of options and amounts is equal before continuing
		if ( count( $options ) !== count( $amounts ) ) {
			return (
				'<div style="color: red;">' .
					esc_html__( 'You have a mismatched number of options and amounts. Please correct this for the dropdown to appear.', 'simple-pay' ) .
				'</div>'
			);
		}

		$options_markup = '';

		foreach ( $options as $i => $v ) {
			$options_markup .= '<option ' . selected( $v, self::$default, false ) . ' data-amount="' . esc_attr( intval( $amounts[ $i ] ) ) . '">' . esc_html( $v ) . '</option>';
		}

		return (
			'<select name="' . esc_attr( self::$name ) . '" id="' . esc_attr( self::$id ) . '" class="simpay-amount-dropdown">' .
				$options_markup .
			'</select>'
		);
	}

}
