<?php
/**
 * Forms field: Total Amount
 *
 * @package SimplePay\Pro\Forms\Fields
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.0.0
 */

namespace SimplePay\Pro\Forms\Fields;

use SimplePay\Core\Abstracts\Custom_Field;
use SimplePay\Pro\Payments\Subscription;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Total_Amount class.
 *
 * @since 3.0.0
 */
class Total_Amount extends Custom_Field {

	/**
	 * Prints HTML for field on frontend.
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_html( $settings ) {
		$tax_amount = isset( $settings['tax_amount'] ) && 'yes' === $settings['tax_amount'];
		$recurring  = isset( $settings['recurring_total'] ) && 'yes' === $settings['recurring_total'];

		ob_start();

		echo '<div class="simpay-form-control">';

		// Tax label.
		if ( true === $tax_amount && self::$form->tax_percent > 0 ) {
			self::print_tax_amount_label( $settings );
		}

		// Total label.
		self::print_total_amount_label( $settings );

		// Recurring label.
		if ( true === $recurring && self::$form->is_subscription() ) {
			self::print_recurring_total_label( $settings );
		}

		echo '</div>';

		return ob_get_clean();
	}

	/**
	 * HTML for the total amount label.
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_total_amount_label( $settings ) {
		$label = isset( $settings['label'] ) && ! empty( $settings['label'] )
			? $settings['label']
			: esc_html__( 'Total Amount:', 'simple-pay' );
		?>

		<div class="simpay-total-amount-container">
			<p class="simpay-total-amount-label simpay-label-wrap">
				<?php echo esc_html( $label ); ?>
				<span class="simpay-total-amount-value">
					<?php echo simpay_format_currency( 0, simpay_get_setting( 'currency' ) ); ?>
				</span>
			</p>
		</div>

		<?php
	}

	/**
	 * HTML for the recurring total label
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_recurring_total_label( $settings ) {
		$label = isset( $settings['recurring_total_label'] ) && ! empty( $settings['recurring_total_label'] )
			? $settings['recurring_total_label']
			: esc_html__( 'Recurring Total:', 'simple-pay' );
		?>

		<div class="simpay-total-amount-recurring-container">
			<p class="simpay-total-amount-recurring-label simpay-label-wrap">
				<?php echo esc_html( $label ); ?>

				<span class="simpay-total-amount-recurring-value">
				<?php
				if ( self::$form->has_subscription_custom_amount && ( self::$form->subscription_interval > 1 ) ) :
					echo esc_html(
						sprintf(
							/** translators: %1$s Recurring total amount. %2$s Recurring interval. %3$s Recurring frequency. */
							__( '%1$s every %2$s %3$s', 'simple-pay' ),
							simpay_format_currency( 0, simpay_get_setting( 'currency' ) ),
							self::$form->subscription_interval,
							strtolower(
								Subscription\get_subscription_interval_label(
									self::$form->subscription_frequency,
									'plural'
								)
							)
						)
					);
				else :
					echo esc_html(
						sprintf(
							/** translators: %1$s Recurring total amount. %2$s Recurring frequency. */
							__( '%1$s/%2$s', 'simple-pay' ),
							simpay_format_currency( 0, simpay_get_setting( 'currency' ) ),
							strtolower(
								Subscription\get_subscription_interval_label(
									self::$form->subscription_frequency,
									'singular'
								)
							)
						)
					);
				endif;
				?>
				</span>
			</p>
		</div>

		<?php
	}

	/**
	 * HTML for the tax amount label
	 *
	 * @since 3.0.0
	 *
	 * @param array $settings Field settings.
	 * @return string
	 */
	public static function print_tax_amount_label( $settings ) {
		$label = isset( $settings['tax_amount_label'] ) && ! empty( $settings['tax_amount_label'] )
			? $settings['tax_amount_label']
			: esc_html__( 'Tax Amount:', 'simple-pay' );
		?>

		<div class="simpay-tax-amount-container">
			<p class="simpay-total-amount-tax-label simpay-label-wrap">
				<?php echo esc_html( $label ); ?>
				<span class="simpay-tax-amount-value">
					<?php echo simpay_format_currency( 0 ); ?>
				</span>
			</p>
		</div>

		<?php
	}
}
