<?php
/**
 * Payment Methods: Payment Method
 *
 * @package SimplePay\Core\Payment_Methods
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Payment_Methods;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Payment_Method {

	/**
	 * Payment Method ID.
	 *
	 * @since 3.8.0
	 * @var string
	 */
	public $id;

	/**
	 * Payment Method name.
	 *
	 * @since 3.8.0
	 * @var string
	 */
	public $name;

	/**
	 * Payment Method nicename.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	public $nicename;

	/**
	 * Payment Method flow.
	 *
	 * @since 3.8.0
	 * @var string
	 */
	public $flow;

	/**
	 * Payment Method countries.
	 *
	 * @since 3.8.0
	 * @var array
	 */
	public $countries;

	/**
	 * Payment Method currencies.
	 *
	 * @since 3.8.0
	 * @var array
	 */
	public $currencies;

	/**
	 * Payment Method recurring support.
	 *
	 * @since 3.8.0
	 * @var bool
	 */
	public $recurring;

	/**
	 * Payment Method Stripe Checkout support.
	 *
	 * @since 3.8.0
	 * @var bool
	 */
	public $stripe_checkout;

	/**
	 * Internal (vs. something like Stripe) documentation URL.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	public $internal_docs;

	/**
	 * External (vs. in-house) documentation URL.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	public $external_docs;

	/**
	 * Constructs the Payment Method.
	 *
	 * @since 3.8.0
	 *
	 * @param array $args {
	 *   Payment Method configuration.
	 *
	 *   @type string $id          Payment Method identifier.
	 *   @type string $name        Payment Method name.
	 *   @type string $flow        Payment Method flow. Accepts `none`, `redirect`, `receiver`. Default `none`.
	 *   @type array  $countries   Payment Method country availability. An empty list falls back
	 *                             to Stripe's available countries. Default empty.
	 *   @type array  $currrencies Payment Method available currencies. An empty list falls back
	 *                             to Stripe's available currencies. Default empty.
	 * }
	 */
	public function __construct( $args ) {
		$defaults = array(
			'id'              => '',
			'name'            => '',
			'nicename'        => '',
			'flow'            => 'none',
			'countries'       => array(),
			'currencies'      => array_map( 'strtolower', array_keys( simpay_get_currencies() ) ),
			'recurring'       => false,
			'stripe_checkout' => false,
			'internal_docs'   => '',
			'external_docs'   => '',
		);

		$args = wp_parse_args( $args, $defaults );

		$this->id = sanitize_text_field( $args['id'] );

		$this->name     = sanitize_text_field( $args['name'] );
		$this->nicename = isset( $args['nicename'] )
			? sanitize_text_field( $args['nicename'] )
			: $this->name;

		if ( in_array( $args['flow'], array( 'none', 'redirect', 'receiver' ), true ) ) {
			$this->flow = sanitize_text_field( $args['flow'] );
		}

		if ( is_array( $args['countries'] ) ) {
			$this->countries = array_map( 'sanitize_text_field', $args['countries'] );
			$this->countries = array_map( 'strtolower', $this->countries );
		}

		if ( is_array( $args['currencies'] ) ) {
			$this->currencies = array_map( 'sanitize_text_field', $args['currencies'] );
			$this->currencies = array_map( 'strtolower', $this->currencies );
		}

		$this->recurring = (bool) $args['recurring'];

		$this->stripe_checkout = (bool) $args['stripe_checkout'];

		if ( ! empty( $args['internal_docs'] ) ) {
			$this->internal_docs = esc_url( $args['internal_docs'] );
		}

		if ( ! empty( $args['external_docs'] ) ) {
			$this->external_docs = esc_url( $args['external_docs'] );
		}
	}

	/**
	 * Determines if the Payment Method is available to use.
	 *
	 * @since 3.8.0
	 *
	 * @return bool True if the Payment Method can be used, otherwise false.
	 */
	public function is_available() {
		$is_available = (
			$this->is_currency_supported()
		);

		/**
		 * Filters if the Payment Method is available to use.
		 *
		 * @since 3.8.0
		 *
		 * @param bool                                          $is_available   If the Payment Method is available to use.
		 * @param \SimplePay\Pro\Payment_Methods\Payment_Method $payment_method Payment Method.
		 */
		$is_available = apply_filters( 'simpay_payment_method_is_available', $is_available, $this );

		return $is_available;
	}

	/**
	 * Determines if the Payment Method's available currencies
	 * include the global currency setting.
	 *
	 * @since 3.8.0
	 *
	 * @param string $currency Optional. Currency to check support for. Default site currency.
	 * @return bool True if the site's currency is supported by the Payment Method, otherwise false.
	 */
	public function is_currency_supported( $currency = '' ) {
		$currencies = $this->currencies;
		$currency   = empty( $currency )
			? simpay_get_setting( 'currency' )
			: (string) $currency;

		return in_array( strtolower( $currency ), $currencies, true );
	}
}
