<?php
/**
 * Payment Methods: Functions
 *
 * @package SimplePay\Core\Payment_Methods
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Payment_Methods;
use SimplePay\Core\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Registers available Payment Methods.
 *
 * @since 3.8.0
 *
 * @param \SimplePay\Core\Utils\Collections $registry Collections registry.
 */
function register_payment_methods( $registry ) {
	// Add Payment Methods registry to Collections registry.
	$payment_methods = new Collection();
	$registry->add( 'payment-methods', $payment_methods );

	// Card.
	$card = new Payment_Method(
		array(
			'id'              => 'card',
			'name'            => esc_html__( 'Card', 'simple-pay' ),
			'nicename'        => esc_html__( 'Credit Card', 'simple-pay' ),
			'recurring'       => true,
			'stripe_checkout' => true,
			'external_docs'   => 'https://stripe.com/payments/payment-methods-guide#cards',
		)
	);

	$payment_methods->add( 'card', $card );

	// ACH (via Plaid).
	$ach_debit = new Payment_Method(
		array(
			'id'              => 'ach-debit',
			'name'            => esc_html__( 'ACH Debit (Direct Debit Payments)', 'simple-pay' ),
			'nicename'        => esc_html__( 'Direct Debit', 'simple-pay' ),
			'recurring'       => true,
			'stripe_checkout' => false,
			'currencies'      => array(
				'usd',
			),
			'external_docs'   => 'https://stripe.com/payments/payment-methods-guide#ach-debits',
		)
	);

	$payment_methods->add( 'ach-debit', $ach_debit );

	// iDEAL.
	$ideal = new Payment_Method(
		array(
			'id'              => 'ideal',
			'name'            => esc_html__( 'iDEAL', 'simple-pay' ),
			'nicename'        => esc_html__( 'iDEAL', 'simple-pay' ),
			'recurring'       => false,
			'stripe_checkout' => true,
			'currencies'      => array(
				'eur',
			),
			'external_docs'   => 'https://stripe.com/payments/payment-methods-guide#ideal',
		)
	);

	$payment_methods->add( 'ideal', $ideal );

	/**
	 * Allows further Payment Methods to be registered.
	 *
	 * @since 3.8.0
	 *
	 * @param \SimplePay\Core\Utils\Collection $payment_methods Payment Methods registry.
	 */
	do_action( 'simpay_register_payment_methods', $payment_methods );
}
add_action( 'simpay_register_collections', __NAMESPACE__ . '\\register_payment_methods' );

/**
 * Returns a list of registered Payment Methods.
 *
 * @since 3.8.0
 *
 * @return array List of Payment Methods.
 */
function get_payment_methods() {
	$payment_methods = Utils\get_collection( 'payment-methods' );

	if ( false === $payment_methods ) {
		return array();
	}

	return $payment_methods->get_items();
}

/**
 * Returns a Payment Method.
 *
 * @since 3.8.0
 *
 * @param string $payment_method_id ID of the registered Payment Method.
 * @return false|\SimplePay\Core\Payment_Methods\Payment_Method Payment Method if found, otherwise `false`.
 */
function get_payment_method( $payment_method_id ) {
	$payment_methods = Utils\get_collection( 'payment-methods' );

	if ( false === $payment_methods ) {
		return false;
	}

	return $payment_methods->get_item( $payment_method_id );
}

/**
 * Returns a list of registered Payment Methods that
 * support Stripe Checkout.
 *
 * @since 3.8.0
 *
 * @return array List of Payment Methods that support Stripe Checkout.
 */
function get_stripe_checkout_payment_methods() {
	$payment_methods = Utils\get_collection( 'payment-methods' );

	if ( false === $payment_methods ) {
		return array();
	}

	return array_filter(
		$payment_methods->get_items(),
		function( $payment_method ) {
			return true === $payment_method->stripe_checkout;
		}
	);
}

/**
 * Retrieves saved Payment Methods for a specific form.
 *
 * @since 3.8.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Payment Form.
 * @return SimplePay\Pro\Payment_Methods\Payment_Method[] List of Payment Methods.
 */
function get_form_payment_methods( $form ) {
	$payment_form = 'stripe_checkout' === $form->get_form_display_type()
		? 'stripe-checkout'
		: 'stripe-elements';

	$payment_methods = simpay_get_filtered(
		'payment_methods',
		simpay_get_saved_meta( $form->id, '_payment_methods', array() ),
		$form->id
	);

	// Form hasn't been updated since 3.8.0.
	if ( empty( $payment_methods ) ) {
		$payment_methods = array(
			'card' => array(
				'id' => 'card',
			),
		);
		// Use saved Payment Methods for Payment Form.
	} else {
		$payment_methods = isset( $payment_methods[ $payment_form ] )
			? $payment_methods[ $payment_form ]
			: array();
	}

	$payment_methods = array_map(
		/**
		 * Attach saved Payment Method settings to the \SimplePay\Pro\Payment_Methods\Payment_Method.
		 *
		 * @since 3.8.0
		 *
		 * @param array $payment_method Saved Payment Method data.
		 * @return false|\SimplePay\Pro\Payment_Methods\Paymetn_Method Payment Method object if available
		 *                                                             otherwise false.
		 */
		function( $payment_method ) use ( $form ) {
			if ( ! isset( $payment_method['id'] ) ) {
				return false;
			}

			$payment_method_obj = get_payment_method( $payment_method['id'] );

			if ( false === $payment_method_obj ) {
				return false;
			}

			if ( false === $payment_method_obj->is_currency_supported( $form->currency ) ) {
				return false;
			}

			// Append any additional saved data.
			unset( $payment_method['id'] );

			return $payment_method_obj;
		},
		$payment_methods
	);

	// General sorting/priority.
	// @todo Add sorting in admin UI.
	$payment_methods = array_replace(
		array_flip(
			array(
				'card',
				'ach-debit',
				'ideal',
			)
		),
		$payment_methods
	);

	return array_filter(
		$payment_methods,
		function( $payment_method ) {
			return is_a( $payment_method, 'SimplePay\Pro\Payment_Methods\Payment_Method' );
		}
	);
}


/**
 * Retrieves saved Payment Method settings for a specific form.
 *
 * @since 3.9.0
 *
 * @param SimplePay\Core\Abstracts\Form $form           Payment Form.
 * @param string                        $payment_method Payment Method ID.
 * @return array List of Payment Method settings.
 */
function get_form_payment_method_settings( $form, $payment_method ) {
	$payment_form = 'stripe_checkout' === $form->get_form_display_type()
		? 'stripe-checkout'
		: 'stripe-elements';

	$payment_methods = simpay_get_filtered(
		'payment_methods',
		simpay_get_saved_meta( $form->id, '_payment_methods', array() ),
		$form->id
	);

	// Form hasn't been updated since 3.8.0.
	if ( empty( $payment_methods ) ) {
		$payment_methods = array(
			'card' => array(
				'id' => 'card',
			),
		);
		// Use saved Payment Methods for Payment Form.
	} else {
		$payment_methods = isset( $payment_methods[ $payment_form ] )
			? $payment_methods[ $payment_form ]
			: array();
	}

	if ( ! isset( $payment_methods[ $payment_method ] ) ) {
		return array();
	}

	return $payment_methods[ $payment_method ];
}
