<?php
/**
 * Payments: Shared
 *
 * @package SimplePay\Pro\Payments
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Stores General field values in object metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args Object arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 */
function add_general_metadata( $object_args, $form, $form_data, $form_values ) {
	// Can't do anything without custom fields.
	if ( ! isset( $form_values['simpay_field'] ) || empty( $form_values['simpay_field'] ) ) {
		return $object_args;
	}

	foreach ( $form_values['simpay_field'] as $name => $value ) {
		// Skip empty.
		if ( '' === $value ) {
			continue;
		}

		$name  = simpay_truncate_metadata( 'title', $name );
		$value = simpay_truncate_metadata( 'description', $value );

		$name  = sanitize_text_field( stripslashes( $name ) );
		$value = sanitize_text_field( stripslashes( $value ) );

		$object_args['metadata'][ $name ] = $value;
	}

	return $object_args;
}
add_filter( 'simpay_get_paymentintent_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );
add_filter( 'simpay_get_subscription_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );
add_filter( 'simpay_get_charge_args_from_payment_form_request', __NAMESPACE__ . '\\add_general_metadata', 10, 4 );

/**
 * Stores coupon information in PaymentIntent metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args PaymentIntent arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return array
 */
function add_coupon_metadata( $object_args, $form, $form_data, $form_values ) {
	$coupon = isset( $form_values['simpay_coupon'] ) ? esc_html( $form_values['simpay_coupon'] ) : false;

	if ( false === $coupon ) {
		return $object_args;
	}

	$coupon_details = isset( $form_values['simpay_coupon_details'] ) ? esc_html( $form_values['simpay_coupon_details'] ) : false;

	if ( $coupon_details ) {
		$value = simpay_truncate_metadata( 'description', $coupon_details );
	} else {
		$value = simpay_truncate_metadata( 'description', $coupon );
	}

	$object_args['metadata']['coupon_code'] = $value;

	return $object_args;
}
add_filter( 'simpay_get_paymentintent_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );
add_filter( 'simpay_get_subscription_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );
add_filter( 'simpay_get_charge_args_from_payment_form_request', __NAMESPACE__ . '\\add_coupon_metadata', 10, 4 );

/**
 * Stores tax information in PaymentIntent metadata.
 *
 * @since 3.6.0
 *
 * @param array                         $object_args PaymentIntent arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return array
 */
function add_tax_metadata( $object_args, $form, $form_data, $form_values ) {
	$tax_amount = isset( $form_values['simpay_tax_amount'] ) ? floatval( $form_values['simpay_tax_amount'] ) : 0;

	if ( 0 === $tax_amount ) {
		return $object_args;
	}

	$currency = isset( $object_args['currency'] ) ? $object_args['currency'] : $form->currency;

	$object_args['metadata']['simpay_tax_amount']  = html_entity_decode(
		simpay_format_currency( $tax_amount, $currency )
	);
	$object_args['metadata']['simpay_tax_percent'] = $form->tax_percent;

	return $object_args;
}
add_filter( 'simpay_get_paymentintent_args_from_payment_form_request', __NAMESPACE__ . '\\add_tax_metadata', 10, 4 );
add_filter( 'simpay_get_subscription_args_from_payment_form_request', __NAMESPACE__ . '\\add_tax_metadata', 10, 4 );
add_filter( 'simpay_get_charge_args_from_payment_form_request', __NAMESPACE__ . '\\add_tax_metadata', 10, 4 );
