<?php
/**
 * Stripe Checkout: Subscription
 *
 * @package SimplePay\Pro\Payments\Stripe_Checkout\Subscription
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments\Stripe_Checkout\Subscription;

use SimplePay\Core\Legacy;
use SimplePay\Pro\Payments\Subscription;
use SimplePay\Pro\Payments\Plan;
use SimplePay\Pro\Payments\Product;
use SimplePay\Pro\Payments\Stripe_Checkout;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds `subscription_data` to Stripe Checkout Session.
 *
 * @since 3.6.0
 *
 * @param array                         $session_args Arguments used to create a PaymentIntent.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function add_subscription( $session_args, $form, $form_data, $form_values, $customer_id ) {
	// Single charges need nothing changed.
	if ( ! Subscription\payment_form_request_has_subscription( $form, $form_data, $form_values ) ) {
		return $session_args;
	}

	// Remove/reset arguments that are not supported by Checkout.
	// @see SimplePay\Core\SimplePay\Core\Stripe_Checkout\Session\get_args_from_payment_form_request()
	unset( $session_args['line_items'] );
	unset( $session_args['payment_intent_data'] );
	unset( $session_args['subscription_data']['tax_percent'] );

	if ( isset( $session_args['submit_type'] ) ) {
		unset( $session_args['submit_type'] );
	}

	// Generate subscription data from payment form.
	$session_args['subscription_data'] = get_subscription_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id );

	return $session_args;
}
add_filter( 'simpay_get_session_args_from_payment_form_request', __NAMESPACE__ . '\\add_subscription', 10, 5 );

/**
 * Uses the current payment form request to generate arguments for a Checkout Session's subscription_data.
 *
 * This differs from (with some duplication unfortunately) from
 * `SimplePay\Pro\Payments\Subscription\get_args_from_payment_form_request()`
 * because it uses an existing plan to generate a Product + Plan combination that will
 * worth the restrictions imposed by Stripe Checkout when dealing with Subscriptions.
 *
 * Currently Subscriptions in Checkout Sessions require:
 *
 * - Only using `trial_period_days` at the Subscription level. Trying to subscribe to a Plan with
 *   `trial_period_days` set will result in an error.
 * - Items cannot have `tax_percent`, nor can `tax_percent` be set on the Subscription level.
 *   Instead tax amounts are added to the Setup Fee line item amount and Plan amount to
 *   create a new total amount for each.
 *
 * @see SimplePay\Pro\Payments\Subscription\get_args_from_payment_form_request()
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function get_subscription_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
	$subscription_args = array(
		'items'    => array(),
		'metadata' => array_merge(
			Legacy\Hooks\simpay_payment_metadata( $form, $form_data, $form_values, $customer_id ),
			array(
				'simpay_form_id'          => $form->id,
				'simpay_subscription_key' => Subscription\get_subscription_key( $customer_id ),
				'description'             => Legacy\Hooks\simpay_payment_description(
					$form,
					$form_data,
					$form_values,
					$customer_id
				),
			)
		),
	);

	if ( ! Plan\payment_form_request_needs_custom_plan( $form, $form_data, $form_values ) ) {
		// Plan based off a selected plan.
		if ( 'user' === $form->subscription_type ) {
			$plan_template = Plan\retrieve(
				$form_values['simpay_multi_plan_id'],
				$form->get_api_request_args()
			);
			// Plan based off a single set plan.
		} elseif ( 'single' === $form->subscription_type ) {
			$plan_template = Plan\retrieve(
				$form->single_plan,
				$form->get_api_request_args()
			);
		}

		$product = Product\retrieve(
			$plan_template->product,
			$form->get_api_request_args()
		);
		// Plan based off a custom amount, stub a template.
	} else {
		$plan_template                    = new \stdClass();
		$plan_template->id                = 'plan_' . uniqid();
		$plan_template->currency          = $form->currency;
		$plan_template->trial_period_days = 0;

		// Amount set via "Custom Amount" + "Recurring Amount Toggle".
		if ( Subscription\payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) ) {
			if ( isset( $form_values['simpay_custom_amount'] ) ) {
				$plan_template->amount = simpay_convert_amount_to_cents( $form_values['simpay_custom_amount'] );
			} else {
				$plan_template->amount = simpay_convert_amount_to_cents( $form->amount );
			}

			$plan_template->interval       = $form->recurring_amount_toggle_frequency;
			$plan_template->interval_count = $form->recurring_amount_toggle_interval;

			// Amount set via Subscription "Custom Amount" field.
		} else {
			$plan_template->amount         = simpay_convert_amount_to_cents( $form_values['simpay_subscription_custom_amount'] );
			$plan_template->interval       = $form->subscription_frequency;
			$plan_template->interval_count = $form->subscription_interval;
		}

		$plan_template->nickname = Plan\generate_product_name(
			simpay_format_currency( simpay_convert_amount_to_dollars( $plan_template->amount ), $form->currency ),
			$plan_template->interval_count,
			$plan_template->interval
		);

		$product = null;
	}

	// Create a custom plan based on the selected predefined plan, or custom amount.
	$plan = create_custom_plan_from_template(
		$plan_template,
		$form,
		$form_data,
		$form_values,
		$customer_id,
		$product
	);

	$subscription_args['items'][0]['plan'] = $plan->id;

	// Add quantity.
	$subscription_args['items'][0]['quantity'] = intval( isset( $form_values['simpay_quantity'] ) ? $form_values['simpay_quantity'] : 1 );

	// Add trial length if needed.
	if ( $plan_template->trial_period_days ) {
		$subscription_args['trial_period_days'] = $plan_template->trial_period_days;
	}

	// Add installment plan metadata if needed.
	//
	// @link https://docs.wpsimplepay.com/articles/installment-plans/
	// @todo This section is the same as Elements subscription and should be combined.
	//
	// Max charges set via "Recurring Amount Toggle" field.
	$max_charges  = 0;
	$charge_count = 0;

	if ( Subscription\payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) ) {
		if ( 0 !== intval( $form->recurring_amount_toggle_max_charges ) ) {
			$max_charges = intval( $form->recurring_amount_toggle_max_charges );
		}

		// Max charges set via Subscription options.
	} else {
		// Custom plan.
		if ( 0 !== intval( $form->subscription_max_charges ) ) {
			$max_charges = intval( $form->subscription_max_charges );
		}

		// Set plan (priority).
		// Ensure it's not 0 to prevent overriding the custom amount max.
		if ( isset( $form_values['simpay_max_charges'] ) && 0 !== intval( $form_values['simpay_max_charges'] ) ) {
			$max_charges = intval( $form_values['simpay_max_charges'] );
		}

		$charge_count = $plan_template->trial_period_days ? - 1 : $charge_count;
	}

	if ( 0 !== $max_charges ) {
		$subscription_args['metadata']['simpay_charge_max']   = $max_charges;
		$subscription_args['metadata']['simpay_charge_count'] = $charge_count;
	}

	/**
	 * Filters arguments used to generate a Subscription from a posted form.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $subscription_args Arguments used to create the Subscription.
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	$subscription_args = apply_filters(
		'simpay_get_subscription_args_from_payment_form_request',
		$subscription_args,
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	return $subscription_args;
}

/**
 * Creates a custom Plan based on a Plan previously created in the Stripe Dashboard.
 *
 * @since 3.6.0
 * @since 3.7.0 Moved to SimplePay\Pro\Payments\Stripe_Checkout\Plan
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @param \Stripe\Plan|null             $product Attached Plan product if using a template, or null.
 */
function create_custom_plan_from_template( $plan_template, $form, $form_data, $form_values, $customer_id, $product = null ) {
	return Stripe_Checkout\Plan\create_custom_plan_from_template( $plan_template, $form, $form_data, $form_values, $customer_id, $product );
}

/**
 * Adds a setup fee to the Subscription by adding an line_items to the Stripe Checkout Session.
 *
 * @since 3.6.0
 *
 * @param array                         $session_args Checkout Session arguments.
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function add_setup_fee_line_item( $session_args, $form, $form_data, $form_values, $customer_id ) {
	// Single charges cannot have setup fees.
	if ( ! Subscription\payment_form_request_has_subscription( $form, $form_data, $form_values ) ) {
		return $session_args;
	}

	// Plan fee.
	$plan_fee = isset( $form_values['simpay_multi_plan_setup_fee'] )
		? simpay_unformat_currency( $form_values['simpay_multi_plan_setup_fee'] )
		: 0;

	if ( $plan_fee > 0 ) {
		if ( $form->tax_percent ) {
			$plan_fee += simpay_calculate_tax_amount( $plan_fee, $form->tax_percent );
		}

		$plan_fee_args = array(
			'name'     => __( 'Subscription Setup Fee', 'simple-pay' ),
			'amount'   => simpay_convert_amount_to_cents( $plan_fee ),
			'currency' => $form->currency,
			'quantity' => 1,
		);

		/** This filter is documented in includes/pro/payments/subscription.php */
		$plan_fee_args = apply_filters(
			'simpay_get_plan_setup_fee_args_from_payment_form_request',
			$plan_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		$session_args['line_items'][] = $plan_fee_args;
	}

	// Initial fee.
	$initial_fee = $form->subscription_setup_fee;

	if ( $initial_fee > 0 ) {
		if ( $form->tax_percent ) {
			$initial_fee += simpay_calculate_tax_amount( $initial_fee, $form->tax_percent );
		}

		$initial_fee_args = array(
			'name'     => __( 'Initial Setup Fee', 'simple-pay' ),
			'amount'   => simpay_convert_amount_to_cents( $initial_fee ),
			'currency' => $form->currency,
			'quantity' => 1,
		);

		/** This filter is documented in includes/pro/payments/subscription.php */
		$initial_fee_args = apply_filters(
			'simpay_get_initial_setup_fee_args_from_payment_form_request',
			$initial_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		$session_args['line_items'][] = $initial_fee_args;
	}

	return $session_args;
}
add_action( 'simpay_get_session_args_from_payment_form_request', __NAMESPACE__ . '\\add_setup_fee_line_item', 20, 5 );
