<?php
/**
 * Subscription functionality.
 *
 * @since 3.6.0
 */

namespace SimplePay\Pro\Payments\Subscription;

use SimplePay\Core\Payments\Stripe_API;
use SimplePay\Core\Legacy;
use SimplePay\Pro\Payments;
use SimplePay\Pro\Payments\Plan;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Retrieves a Subscription.
 *
 * @since 3.7.0
 *
 * @param string|array $subscription Subscription ID or {
 *   Arguments used to retrieve a Subscription.
 *
 *   @type string $id Subscription ID.
 * }
 * @param array        $api_request_args {
 *         Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\Subscription
 * @throws \Stripe\Exception
 */
function retrieve( $subscription, $api_request_args = array() ) {
	if ( false === is_array( $subscription ) ) {
		$subscription_args = array(
			'id' => $subscription,
		);
	} else {
		$subscription_args = $subscription;
	}

	return Stripe_API::request(
		'Subscription',
		'retrieve',
		$subscription_args,
		$api_request_args
	);
}

/**
 * Retrieves Subscriptions.
 *
 * @since 3.8.0
 *
 * @param array $subscriptions Optional arguments used when listing Subscriptions.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return object
 */
function all( $subscriptions = array(), $api_request_args = array() ) {
	return Stripe_API::request(
		'Subscription',
		'all',
		$subscriptions,
		$api_request_args
	);
}

/**
 * Creates a Subscription.
 *
 * @since 3.6.0
 *
 * @param array $subscription_args Optional arguments used to create a subscription.
 * @param array $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\Subscription
 * @throws \Stripe\Exception
 */
function create( $subscription_args = array(), $api_request_args = array() ) {
	$defaults          = array();
	$subscription_args = wp_parse_args( $subscription_args, $defaults );

	/**
	 * Filter the arguments passed to Subscription creation in Stripe.
	 *
	 * @since 3.6.0
	 *
	 * @param array $subscription_args Arguments passed to Subscription creation in Stripe.
	 */
	$subscription_args = apply_filters( 'simpay_create_stripe_subscription_args', $subscription_args );

	/**
	 * Allows processing before a Subscription is created.
	 *
	 * @since 3.6.0
	 *
	 * @param array $subscription_args Optional arguments used to create a Subscription.
	 */
	do_action( 'simpay_before_subscription_created', $subscription_args );

	$subscription = Stripe_API::request(
		'Subscription',
		'create',
		$subscription_args,
		$api_request_args
	);

	/**
	 * Allows further processing after a Subscription has been created.
	 *
	 * @since 3.6.0
	 *
	 * @param \Stripe\Subscription $subscription Subscription.
	 */
	do_action( 'simpay_after_subscription_created', $subscription );

	move_metadata_to_linked_objects( $subscription );

	return $subscription;
}

/**
 * Updates a Subscription record.
 *
 * @since 3.7.0
 *
 * @param string $subscription_id ID of the Customer to update.
 * @param array  $subscription_args Data to update Customer with.
 * @param array  $api_request_args {
 *   Additional request arguments to send to the Stripe API when making a request.
 *
 *   @type string $api_key API Secret Key to use.
 * }
 * @return \Stripe\Customer $subscription Stripe Customer.
 * @throws \Stripe\Exception
 */
function update( $subscription_id, $subscription_args, $api_request_args = array() ) {
	/**
	 * Filters the arguments passed to Subscription creation in Stripe.
	 *
	 * @since 3.7.0
	 *
	 * @param array $subscription_args Arguments passed to subscription creation in Stripe.
	 * @param string $subscription_id ID of the Customer to update.
	 */
	$subscription_args = apply_filters(
		'simpay_update_subscription_args',
		$subscription_args,
		$subscription_id
	);

	$subscription = Stripe_API::request(
		'Subscription',
		'update',
		$subscription_id,
		$subscription_args,
		$api_request_args
	);

	/**
	 * Allows further processing after a Subscription has been update.
	 *
	 * @since 3.7.0
	 *
	 * @param \Stripe\Subscription $subscription Stripe Subscription.
	 */
	do_action( 'simpay_after_subscription_created', $subscription );

	return $subscription;
}

/**
 * Generate a unique Subscription key based on a Customer ID.
 *
 * @since 3.7.0
 *
 * @param string $customer_id Customer ID.
 * @return string
 */
function get_subscription_key( $customer_id ) {
	$auth_key         = defined( 'AUTH_KEY' ) ? AUTH_KEY : '';
	$hash             = $customer_id . date( 'Y-m-d H:i:s' ) . $auth_key . uniqid( 'wpsp', true );
	$subscription_key = strtolower( md5( $hash ) );

	return $subscription_key;
}

/**
 * Uses the current payment form request to generate arguments for a Subscription.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 * @return array
 */
function get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
	$subscription_args = array(
		'items'    => array(),
		'metadata' => array_merge(
			Legacy\Hooks\simpay_payment_metadata( $form, $form_data, $form_values, $customer_id ),
			array(
				'simpay_form_id'          => $form->id,
				'simpay_subscription_key' => get_subscription_key( $customer_id ),
				'description'             => Legacy\Hooks\simpay_payment_description(
					$form,
					$form_data,
					$form_values,
					$customer_id
				),
			)
		),
	);

	// Custom amount, need to create a Plan in Stripe.
	if ( Payments\Plan\payment_form_request_needs_custom_plan( $form, $form_data, $form_values ) ) {
		$plan = create_custom_subscription_plan_from_payment_form_request( $form, $form_data, $form_values, $customer_id );
	} else {
		if ( 'user' === $form->subscription_type ) {
			$plan = Plan\retrieve(
				$form_values['simpay_multi_plan_id'],
				$form->get_api_request_args()
			);
		} elseif ( 'single' === $form->subscription_type ) {
			$plan = Plan\retrieve(
				$form->single_plan,
				$form->get_api_request_args()
			);
		}
	}

	$subscription_args['items'][0]['plan'] = $plan->id;

	// Add quantity.
	$subscription_args['items'][0]['quantity'] = intval( isset( $form_values['simpay_quantity'] ) ? $form_values['simpay_quantity'] : 1 );

	// Add tax_percent if needed.
	//
	// @todo This might be removed in future versions of the Stripe API.
	//
	// @link https://stripe.com/docs/api/subscriptions/create#create_subscription-tax_percent
	// @link https://stripe.com/docs/billing/migration/taxes
	if ( $form->tax_percent ) {
		$subscription_args['tax_percent'] = $form->tax_percent;
	}

	// Add installment plan metadata if needed.
	//
	// @link https://docs.wpsimplepay.com/articles/installment-plans/
	//
	// Max charges set via "Recurring Amount Toggle" field.
	$max_charges  = 0;
	$charge_count = 0;

	if ( payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) ) {
		if ( 0 !== intval( $form->recurring_amount_toggle_max_charges ) ) {
			$max_charges = intval( $form->recurring_amount_toggle_max_charges );
		}

		// Max charges set via Subscription options.
	} else {
		// Custom plan.
		if ( 0 !== intval( $form->subscription_max_charges ) ) {
			$max_charges = intval( $form->subscription_max_charges );
		}

		// Set plan (priority).
		// Ensure it's not 0 to prevent overriding the custom amount max.
		if ( isset( $form_values['simpay_max_charges'] ) && 0 !== intval( $form_values['simpay_max_charges'] ) ) {
			$max_charges = intval( $form_values['simpay_max_charges'] );
		}

		$charge_count = $plan->trial_period_days ? -1 : 0;
	}

	if ( 0 !== $max_charges ) {
		$subscription_args['metadata']['simpay_charge_max']   = $max_charges;
		$subscription_args['metadata']['simpay_charge_count'] = $charge_count;
	}

	// Add trial length if needed.
	//
	// This could be inherited from the Plan, but Stripe prefers setting an end
	// date on each subscription.
	//
	// @link https://stripe.com/docs/api/subscriptions/create#create_subscription-trial_period_days
	if ( $plan->trial_period_days ) {
		$subscription_args['trial_period_days'] = $plan->trial_period_days;
	}

	/**
	 * Filters arguments used to generate a Subscription from a payment form request.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $subscription_args Arguments used to create the Subscription.
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	$subscription_args = apply_filters(
		'simpay_get_subscription_args_from_payment_form_request',
		$subscription_args,
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	return $subscription_args;
}

/**
 * Creates a custom Plan based on the Payment Form's subscription settings.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 */
function create_custom_subscription_plan_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
	$currency = strtolower( $form_data['stripeParams']['currency'] );

	if ( empty( $customer_id ) ) {
		$generated_suffix = esc_html__( ' - generated by WP Simple Pay', 'simple-pay' );
	} else {
		/** translators: %1$s Stripe Customer ID */
		$generated_suffix = sprintf( esc_html__( ' - generated by WP Simple Pay for %1$s', 'simple-pay' ), $customer_id );
	}

	/** This filter is documented in includes/pro/payments/stripe-checkout/subscription.php */
	$generated_suffix = apply_filters(
		'simpay_stripe_checkout_generated_plan_suffix',
		$generated_suffix,
		$form,
		$form_data,
		$form_values,
		$customer_id,
		null
	);

	$id = uniqid();

	$custom_plan_args = array(
		'id'       => $id,
		'nickname' => $id . $generated_suffix,
		'currency' => $currency,
		'metadata' => array(
			'simpay_is_generated_plan' => 1,
		),
	);

	// Amount set via "Custom Amount" + "Recurring Amount Toggle".
	if ( Payments\Subscription\payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) ) {
		if ( isset( $form_values['simpay_custom_amount'] ) ) {
			$amount = simpay_convert_amount_to_cents( $form_values['simpay_custom_amount'] );
		} else {
			$amount = simpay_convert_amount_to_cents( $form->amount );
		}

		$interval       = $form->recurring_amount_toggle_frequency;
		$interval_count = $form->recurring_amount_toggle_interval;

		// Amount set via Subscription "Custom Amount" field.
	} else {
		$amount         = simpay_convert_amount_to_cents( $form_values['simpay_subscription_custom_amount'] );
		$interval       = $form->subscription_frequency;
		$interval_count = $form->subscription_interval;
	}

	$formatted_amount = simpay_format_currency( simpay_convert_amount_to_dollars( $amount ), $currency );

	$product_name = ! empty( $form->company_name )
		? $form->company_name
		: get_bloginfo( 'name' );

	$custom_plan_args['amount']         = $amount;
	$custom_plan_args['interval']       = $interval;
	$custom_plan_args['interval_count'] = $interval_count;
	$custom_plan_args['product']        = array(
		'name'     => $product_name,
		'metadata' => array(
			'simpay_is_generated_product' => 1,
		),
	);

	/**
	 * Filter the arguments used to create the custom Plan for a Subscription.
	 *
	 * @since 3.6.0
	 *
	 * @param array                         $custom_plan_args Arguments used to create the Plan.
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 */
	$custom_plan_args = apply_filters(
		'simpay_get_custom_plan_args_from_payment_form_request',
		$custom_plan_args,
		$form,
		$form_data,
		$form_values,
		$customer_id
	);

	return Payments\Plan\create(
		$custom_plan_args,
		$form->get_api_request_args()
	);
}

/**
 * Adds a setup fee to the Subscription by adding an additional invoice to the Customer.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @param int                           $customer_id Stripe Customer ID.
 */
function add_setup_fee_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
	// Plan fee.
	$plan_fee = isset( $form_values['simpay_multi_plan_setup_fee'] )
		? simpay_unformat_currency( $form_values['simpay_multi_plan_setup_fee'] )
		: 0;

	if ( $plan_fee > 0 ) {
		$plan_fee_args = array(
			'description' => __( 'Subscription Setup Fee', 'simple-pay' ),
			'amount'      => simpay_convert_amount_to_cents( $plan_fee ),
			'currency'    => $form->currency,
			'customer'    => $customer_id,
		);

		/**
		 * Filters the arguments used to create the Plan Setup Fee InvoiceItem.
		 *
		 * @since 3.6.0
		 *
		 * @param array                         $plan_fee_args Arguments used to create the InvoiceItem.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		$setup_fee_args = apply_filters(
			'simpay_get_plan_setup_fee_args_from_payment_form_request',
			$plan_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		Stripe_API::request(
			'InvoiceItem',
			'create',
			$plan_fee_args,
			$form->get_api_request_args()
		);
	}

	// Initial fee.
	$initial_fee = $form->subscription_setup_fee;

	if ( $initial_fee > 0 ) {
		$initial_fee_args = array(
			'description' => __( 'Initial Setup Fee', 'simple-pay' ),
			'amount'      => simpay_convert_amount_to_cents( $initial_fee ),
			'currency'    => $form->currency,
			'customer'    => $customer_id,
		);

		/**
		 * Filters the arguments used to create the Initial Setup Fee InvoiceItem.
		 *
		 * @since 3.6.0
		 *
		 * @param array                         $initial_fee_args Arguments used to create the InvoiceItem.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		$initial_fee_args = apply_filters(
			'simpay_get_initial_setup_fee_args_from_payment_form_request',
			$initial_fee_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);

		Stripe_API::request(
			'InvoiceItem',
			'create',
			$initial_fee_args,
			$form->get_api_request_args()
		);
	}
}

/**
 * Moves metadata attached to the Subscription to other relevant linked objects.
 *
 * This allows a more natural discovery of the metadata when viewing Payments.
 *
 * @since 3.6.0
 *
 * @param \Stripe\Subscription $subscription Stripe Subscription.
 */
function move_metadata_to_linked_objects( $subscription ) {
	$invoice       = $subscription->latest_invoice;
	$paymentintent = $subscription->latest_invoice->payment_intent;

	$metadata = $subscription->toArray()['metadata'];
	unset( $metadata['simpay_charge_max'] );
	unset( $metadata['simpay_charge_count'] );

	$form_id = $subscription->metadata->simpay_form_id;

	if ( empty( $form_id ) ) {
		return;
	}

	/** This filter is documented in includes/core/shortcodes.php */
	$form = apply_filters( 'simpay_form_view', '', $form_id );

	if ( empty( $form ) ) {
		$form = new Default_Form( $form_id );
	}

	if ( $invoice->id ) {
		Stripe_API::request(
			'Invoice',
			'update',
			$invoice->id,
			array(
				'metadata' => $metadata,
			),
			$form->get_api_request_args()
		);
	}

	// PaymentIntent does not exist if trialing.
	// @link https://stripe.com/docs/billing/subscriptions/payment#handling-trial
	if ( $paymentintent ) {
		Stripe_API::request(
			'PaymentIntent',
			'update',
			$paymentintent->id,
			array(
				'metadata' => $metadata,
			),
			$form->get_api_request_args()
		);
	}
}

/**
 * Determines if a posted form needs to handle subscription-related processes.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 * @return bool
 */
function payment_form_request_has_subscription( $form, $form_data, $form_values ) {
	$subscription = $form->is_subscription();
	$recurring    = payment_form_request_has_recurring_toggle( $form, $form_data, $form_values );

	return ( $subscription || $recurring );
}

/**
 * Determines if the Subscription is needed due to "Recurring Amount Toggle" field being checked.
 *
 * @since 3.6.0
 *
 * @param SimplePay\Core\Abstracts\Form $form Form instance.
 * @param array                         $form_data Form data generated by the client.
 * @param array                         $form_values Values of named fields in the payment form.
 */
function payment_form_request_has_recurring_toggle( $form, $form_data, $form_values ) {
	return isset( $form_values['recurring_amount_toggle'] );
}

/**
 * Retrieves a translated label for a Subscription's interval.
 *
 * @param string $interval Billing frequency.
 * @param string $type     Singular or plural label. Accepts `singular` and `plural`
 * @return string
 */
function get_subscription_interval_label( $interval, $type = 'singular' ) {
	$labels = array(
		'day'   => array(
			'singular' => __( 'Day', 'simple-pay' ),
			'plural'   => __( 'Days', 'simple-pay' ),
		),
		'week'  => array(
			'singular' => __( 'Week', 'simple-pay' ),
			'plural'   => __( 'Weeks', 'simple-pay' ),
		),
		'month' => array(
			'singular' => __( 'Month', 'simple-pay' ),
			'plural'   => __( 'Months', 'simple-pay' ),
		),
		'year'  => array(
			'singular' => __( 'Year', 'simple-pay' ),
			'plural'   => __( 'Years', 'simple-pay' ),
		),
	);

	if ( ! isset( $labels[ $interval ], $labels[ $interval ][ $type ] ) ) {
		return $interval;
	}

	return $labels[ $interval ][ $type ];
}
