<?php
/**
 * Simple Pay: Actions
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Actions
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Actions;

use SimplePay\Pro\Payments\Plan;
use SimplePay\Pro\Payment_Methods;
use SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Saves the Payment Form's settings.
 *
 * @since 3.8.0
 *
 * @param int      $post_id Current Payment Form ID.
 * @param \WP_Post $post    Current Payment Form \WP_Post object.
 */
function save( $post_id, $post ) {
	/** This filter is documented in includes/core/shortcodes.php */
	$form = apply_filters( 'simpay_form_view', '', $post_id );

	if ( empty( $form ) ) {
		$form = new Default_Form( $post_id );
	}

	// Allow settings to save depending on the current mode.
	$livemode = isset( $_POST['_livemode'] ) && '' !== $_POST['_livemode']
		? (bool) absint( $_POST['_livemode'] )
		: '';

	$prev_livemode = get_post_meta( $post_id, '_livemode_prev', true );
	$prev_livemode = '' !== $prev_livemode
		? (bool) $prev_livemode
		: '';

	// Payment Methods.

	// Ensure `card` is always enabled.
	$default_payment_methods = array(
		'stripe-checkout' => array(
			'card' => array(
				'id' => 'card',
			),
		),
		'stripe-elements' => array(
			'card' => array(
				'id' => 'card',
			),
		),
	);

	$payment_methods = isset( $_POST['_simpay_payment_methods'] ) ? $_POST['_simpay_payment_methods'] : array();
	$payment_methods = wp_parse_args( $payment_methods, $default_payment_methods );

	foreach ( $payment_methods as $context => $methods ) {
		foreach ( $methods as $method_id => $method ) {
			// Remove any invalid Payment Methods.
			if ( false === Payment_Methods\get_payment_method( $method_id ) ) {
				unset( $payment_methods[ $context ][ $method_id ] );
			}

			// @todo Validate settings.
		}

		if ( empty( $payment_methods[ $context ] ) ) {
			$payment_methods[ $context ] = $default_payment_methods[ $context ];
		}
	}

	update_post_meta( $post_id, '_payment_methods', $payment_methods );

	// Amount type.
	$amount_type = isset( $_POST['_amount_type'] )
		? esc_attr( $_POST['_amount_type'] )
		: 'one_time_set';

	update_post_meta( $post_id, '_amount_type', $amount_type );

	// Form display type.
	$form_display_type = isset( $_POST['_form_display_type'] )
		? sanitize_text_field( $_POST['_form_display_type'] )
		: '';

	update_post_meta( $post_id, '_form_display_type', $form_display_type );

	// Stripe Checkout form styles.
	$enable_form_styles = isset( $_POST['_enable_stripe_checkout_form_styles'] )
		? 'yes'
		: 'no';

	update_post_meta( $post_id, '_enable_stripe_checkout_form_styles', $enable_form_styles );

	// Minimum amount.
	$minimum_amount = isset( $_POST['_minimum_amount'] )
		? sanitize_text_field( $_POST['_minimum_amount'] )
		: (
			false !== get_post_meta( $post_id, '_minimum_amount', true )
				? get_post_meta( $post_id, '_minimum_amount', true )
				: simpay_global_minimum_amount()
		);

	update_post_meta( $post_id, '_minimum_amount', $minimum_amount );

	// Custom amount default.
	$custom_amount_default = isset( $_POST['_custom_amount_default'] )
		? sanitize_text_field( $_POST['_custom_amount_default'] )
		: '';

	update_post_meta( $post_id, '_custom_amount_default', $custom_amount_default );

	// Custom amount label.
	$custom_amount_label = isset( $_POST['_custom_amount_label'] )
		? sanitize_text_field( $_POST['_custom_amount_label'] )
		: '';

	update_post_meta( $post_id, '_custom_amount_label', $custom_amount_label );

	// Plan select field label.
	$form_field_label = isset( $_POST['_plan_select_form_field_label'] )
		? sanitize_text_field( $_POST['_plan_select_form_field_label'] )
		: '';

	update_post_meta( $post_id, '_plan_select_form_field_label', $form_field_label );

	// Display type.
	$form_display_type = isset( $_POST['_form_display_type'] )
		? esc_attr( $_POST['_form_display_type'] )
		: 'embedded';

	update_post_meta( $post_id, '_form_display_type', $form_display_type );

	// Subscription display style.
	$display_style = isset( $_POST['_multi_plan_display'] )
		? esc_attr( $_POST['_multi_plan_display'] )
		: 'radio';

	update_post_meta( $post_id, '_multi_plan_display', $display_style );

	// Subscription Plans.
	$subscription_type = isset( $_POST['_subscription_type'] )
		? esc_attr( $_POST['_subscription_type'] )
		: 'disabled';

	update_post_meta( $post_id, '_subscription_type', $subscription_type );

	// Select Plan (single).
	$single_plan = isset( $_POST['_single_plan'] )
		? sanitize_text_field( $_POST['_single_plan'] )
		: 'empty';


	switch ( $livemode ) {
		case true:
			if ( true === $prev_livemode ) {
				update_post_meta( $post_id, '_single_plan', $single_plan );
			} else {
				update_post_meta( $post_id, '_single_plan_test', $single_plan );
			}

			break;
		case false:
			if ( false === $prev_livemode ) {
				update_post_meta( $post_id, '_single_plan_test', $single_plan );
			} else {
				update_post_meta( $post_id, '_single_plan', $single_plan );
			}

			break;
		default:
			update_post_meta( $post_id, '_single_plan', $single_plan );
	}

	// Plan Setup Fee.
	$setup_fee = isset( $_POST['_setup_fee'] )
		? sanitize_text_field( $_POST['_setup_fee'] )
		: '';

	update_post_meta( $post_id, '_setup_fee', $setup_fee );

	// Max charges.
	$max_charges = isset( $_POST['_max_charges'] )
		? absint( $_POST['_max_charges'] )
		: 0;

	update_post_meta( $post_id, '_max_charges', $max_charges );

	// Custom Plan label.
	$custom_plan_label = isset( $_POST['_custom_plan_label'] )
		? sanitize_text_field( $_POST['_custom_plan_label'] )
		: '';

	update_post_meta( $post_id, '_custom_plan_label', $custom_plan_label );

	// Multi-plans
	$multi_plan = isset( $_POST['_multi_plan'] )
		? $_POST['_multi_plan']
		: array();

	$multi_plan = array_values( $multi_plan );

	// Cache full plan objects from Stripe.
	//
	// @todo This information is available because we had to call `simpay_get_plans()`,
	// so we should send the object through the form POST vs. making further API requests.
	foreach ( $multi_plan as $k => $plan ) {
		try {
			$plan_object = Plan\retrieve(
				$plan['select_plan'],
				$form->get_api_request_args(
					array(
						'livemode' => $prev_livemode,
					)
				)
			);

			$multi_plan[ $k ]['plan_object'] = $plan_object;
		} catch ( \Exception $e ) {
		}
	}

	// Default radio button value for default plan
	$multi_plan_default_value = isset( $_POST['_multi_plan_default_value'] )
		? $_POST['_multi_plan_default_value']
		: '';

	// Convert legacy "empty" string values to the first selected Plan for default.
	if ( 'empty' === $multi_plan_default_value || empty( $multi_plan_default_value ) ) {
		if ( ! empty( $multi_plan ) ) {
			$multi_plan_default_value = current( $multi_plan )['plan_object']->id;
		}
	}

	switch ( $livemode ) {
		case true:
			if ( true === $prev_livemode ) {
				update_post_meta( $post_id, '_multi_plan', $multi_plan );
				update_post_meta( $post_id, '_multi_plan_default_value', $multi_plan_default_value );
			} else {
				update_post_meta( $post_id, '_multi_plan_test', $multi_plan );
				update_post_meta( $post_id, '_multi_plan_default_value_test', $multi_plan_default_value );
			}

			break;
		case false:
			if ( false === $prev_livemode ) {
				update_post_meta( $post_id, '_multi_plan_test', $multi_plan );
				update_post_meta( $post_id, '_multi_plan_default_value_test', $multi_plan_default_value );
			} else {
				update_post_meta( $post_id, '_multi_plan', $multi_plan );
				update_post_meta( $post_id, '_multi_plan_default_value', $multi_plan_default_value );
			}

			break;
		default:
			update_post_meta( $post_id, '_multi_plan', $multi_plan );
			update_post_meta( $post_id, '_multi_plan_default_value', $multi_plan_default_value );
	}

	// Custom amount (multi-plan).
	$subscription_custom_amount = isset( $_POST['_subscription_custom_amount'] )
		? esc_attr( $_POST['_subscription_custom_amount'] )
		: 'disabled';

	update_post_meta( $post_id, '_subscription_custom_amount', $subscription_custom_amount );

	// Minimum amount.
	$multi_plan_minimum_amount = isset( $_POST['_multi_plan_minimum_amount'] )
		? sanitize_text_field( $_POST['_multi_plan_minimum_amount'] )
		: '';

	update_post_meta( $post_id, '_multi_plan_minimum_amount', $multi_plan_minimum_amount );

	// Default amount.
	$multi_plan_default_amount = isset( $_POST['_multi_plan_default_amount'] )
		? sanitize_text_field( $_POST['_multi_plan_default_amount'] )
		: '';

	update_post_meta( $post_id, '_multi_plan_default_amount', $multi_plan_default_amount );

	// Interval.
	$plan_interval = isset( $_POST['_plan_interval'] )
		? intval( $_POST['_plan_interval'] )
		: '';

	update_post_meta( $post_id, '_plan_interval', $plan_interval );

	// Frequency.
	$plan_frequency = isset( $_POST['_plan_frequency'] )
		? esc_attr( $_POST['_plan_frequency'] )
		: '';

	update_post_meta( $post_id, '_plan_frequency', $plan_frequency );

	// Custom fields.
	$fields = isset( $_POST['_simpay_custom_field'] )
		? $_POST['_simpay_custom_field']
		: array();

	// Check & create required missing fields for this form display type.
	$fields = Edit_Form\add_missing_custom_fields( $fields, $post_id, $form_display_type );

	// Re-index the array so if fields were removed we don't overwrite the index with a new field
	foreach ( $fields as $k => $v ) {
		$fields[ $k ] = array_values( $v );
	}

	update_post_meta( $post_id, '_custom_fields', $fields );
}
add_action( 'simpay_save_form_settings', __NAMESPACE__ . '\\save', 10, 2 );
