<?php
/**
 * Simple Pay: Edit form custom fields
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form;

use SimplePay\Pro\Post_Types\Simple_Pay\Util;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Updates the name of the "Custom Fields" tab.
 *
 * In Lite it is named "On-Site Form Display".
 *
 * @since 3.8.0
 *
 * @param array $tabs
 */
function update_setting_tab( $tabs ) {
	$tabs['form_display']['label'] = esc_html__( 'Custom Form Fields', 'simple-pay' );

	return $tabs;
}
add_filter( 'simpay_form_settings_meta_tabs_li', __NAMESPACE__ . '\\update_setting_tab' );

/**
 * Get custom field option group labels.
 *
 * @since 3.8.0
 *
 * @return array Group label names.
 */
function get_custom_field_type_groups() {
	$groups = array(
		'payment'  => _x( 'Payment', 'custom field group', 'simple-pay' ),
		'customer' => _x( 'Customer', 'custom field group', 'simple-pay' ),
		'standard' => _x( 'Standard', 'custom field group', 'simple-pay' ),
		'custom'   => _x( 'Custom', 'custom field group', 'simple-pay' ),
	);

	/**
	 * Filter the labels associated with field groups.
	 *
	 * @since 3.4.0
	 *
	 * @param array $groups optgroup/category keys and associated labels.
	 */
	return apply_filters( 'simpay_custom_field_group_labels', $groups );
}

/**
 * Get the available custom field types.
 *
 * @since 3.8.0
 *
 * @return array $fields Custom fields.
 */
function get_custom_field_types() {
	$fields = array(
		'customer_name'           => array(
			'label'      => esc_html__( 'Name', 'simple-pay' ),
			'type'       => 'customer_name',
			'category'   => 'customer',
			'active'     => true,
			'repeatable' => false,
		),
		'email'                   => array(
			'label'      => esc_html__( 'Email Address', 'simple-pay' ),
			'type'       => 'email',
			'category'   => 'customer',
			'active'     => true,
			'repeatable' => false,
		),
		'telephone'               => array(
			'label'      => esc_html__( 'Phone', 'simple-pay' ),
			'type'       => 'telephone',
			'category'   => 'customer',
			'active'     => true,
			'repeatable' => false,
		),
		'address'                 => array(
			'label'      => esc_html__( 'Address', 'simple-pay' ),
			'type'       => 'address',
			'category'   => 'customer',
			'active'     => true,
			'repeatable' => false,
		),

		'coupon'                  => array(
			'label'      => esc_html__( 'Coupon', 'simple-pay' ),
			'type'       => 'coupon',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => false,
		),
		'custom_amount'           => array(
			'label'      => esc_html__( 'Custom Amount', 'simple-pay' ),
			'type'       => 'custom_amount',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => false,
		),
		'plan_select'             => array(
			'label'      => esc_html__( 'Subscription Plan Selector', 'simple-pay' ),
			'type'       => 'plan_select',
			'category'   => 'payment',
			'active'     => simpay_subscriptions_enabled(),
			'repeatable' => false,
		),
		'recurring_amount_toggle' => array(
			'label'      => esc_html__( 'Recurring Amount Toggle', 'simple-pay' ),
			'type'       => 'recurring_amount_toggle',
			'category'   => 'payment',
			'active'     => simpay_subscriptions_enabled(),
			'repeatable' => false,
		),
		'total_amount'            => array(
			'label'      => esc_html__( 'Total Amount Label', 'simple-pay' ),
			'type'       => 'total_amount',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => true,
		),
		'payment_request_button'  => array(
			'label'      => esc_html__( 'Apple Pay/Google Pay Button', 'simple-pay' ),
			'type'       => 'payment_request_button',
			'category'   => 'payment',
			'active'     => simpay_can_use_payment_request_button(),
			'repeatable' => false,
		),
		'card'                    => array(
			'label'      => esc_html__( 'Payment Methods (Card, ACH, etc)', 'simple-pay' ),
			'type'       => 'card',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => false,
		),
		'checkout_button'         => array(
			'label'      => esc_html__( 'Checkout Button', 'simple-pay' ),
			'type'       => 'checkout_button',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => false,
		),
		'payment_button'          => array(
			'label'      => esc_html__( 'Payment Button', 'simple-pay' ),
			'type'       => 'payment_button',
			'category'   => 'payment',
			'active'     => true,
			'repeatable' => false,
		),

		'heading'                 => array(
			'label'      => esc_html__( 'Heading', 'simple-pay' ),
			'type'       => 'heading',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'text'                    => array(
			'label'      => esc_html__( 'Text', 'simple-pay' ),
			'type'       => 'text',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'dropdown'                => array(
			'label'      => esc_html__( 'Dropdown', 'simple-pay' ),
			'type'       => 'dropdown',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'radio'                   => array(
			'label'      => esc_html__( 'Radio Select', 'simple-pay' ),
			'type'       => 'radio',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'date'                    => array(
			'label'      => esc_html__( 'Date', 'simple-pay' ),
			'type'       => 'date',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'number'                  => array(
			'label'      => esc_html__( 'Number', 'simple-pay' ),
			'type'       => 'number',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'checkbox'                => array(
			'label'      => esc_html__( 'Checkbox', 'simple-pay' ),
			'type'       => 'checkbox',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
		'hidden'                  => array(
			'label'      => esc_html__( 'Hidden', 'simple-pay' ),
			'type'       => 'hidden',
			'category'   => 'standard',
			'active'     => true,
			'repeatable' => true,
		),
	);

	/**
	 * Filters available custom fields.
	 *
	 * @since 3.0.0
	 *
	 * @param array $fields Custom fields.
	 */
	return apply_filters( 'simpay_custom_field_options', $fields );
}

/**
 * Get a grouped list of custom field options.
 *
 * @since 3.8.0
 *
 * @param array $options Flat list of options.
 * @return array $options Grouped list of options.
 */
function get_custom_fields_grouped( $options = array() ) {
	if ( empty( $options ) ) {
		$options = get_custom_field_types();
	}

	$result = array();
	$groups = get_custom_field_type_groups();

	foreach ( $options as $key => $option ) {
		if ( isset( $option['category'] ) ) {
			$result[ $groups[ $option['category'] ] ][ $key ] = $option;
		} else {
			$result[ $groups['custom'] ][ $key ] = $option;
		}
	}

	return $result;
}

/**
 * Adds "Custom Fields" Payment Form settings tab content.
 *
 * @since 3.8.0
 *
 * @param int $post_id Current Payment Form ID.
 */
function add_custom_fields( $post_id ) {
	$field_groups = get_custom_fields_grouped();
	$field_types  = get_custom_field_types();

	if ( empty( $field_groups ) ) {
		return;
	}

	$fields = Util\get_custom_fields( $post_id );

	// Populate an empty form.
	if ( empty( $fields ) ) {
		$fields = array(
			array(
				'type'  => 'email',
				'label' => 'Email Address',
			),
			array(
				'type'  => 'card',
				'label' => 'Payment Method',
			),
			array(
				'type' => 'checkout_button',
			),
		);
	}

	wp_nonce_field( 'simpay_custom_fields_nonce', 'simpay_custom_fields_nonce' );
	?>

<table>
	<tbody class="simpay-panel-section">
		<tr class="simpay-panel-field">
			<th>
				<label for="custom-field-select">
					<?php esc_html_e( 'Form Fields', 'simple-pay' ); ?>
				</label>
			</th>
			<td style="border-bottom: 0;">
				<div class="toolbar toolbar-top">
					<select name="simpay_field_select" id="custom-field-select" class="simpay-field-select">
						<option value=""><?php esc_html_e( 'Choose a field&hellip;', 'simple-pay' ); ?></option>
							<?php foreach ( $field_groups as $group => $options ) : ?>
								<optgroup label="<?php echo esc_attr( $group ); ?>">
									<?php
									foreach ( $options as $option ) :
										if ( ! isset( $option['active'] ) || ! $option['active'] ) :
											continue;
										endif;

										$disabled   = ! isset( $option['repeatable'] ) || ( isset( $fields[ $option['type'] ] ) && ! $option['repeatable'] );
										$repeatable = isset( $option['repeatable'] ) && true === $option['repeatable'];
										?>
										<option
											value="<?php echo esc_attr( $option['type'] ); ?>"
											data-repeatable="<?php echo esc_attr( $repeatable ? 'true' : 'false' ); ?>"
											<?php disabled( true, $disabled ); ?>
										>
											<?php echo esc_html( $option['label'] ); ?>
										</option>
									<?php endforeach; ?>
								</optgroup>
							<?php endforeach; ?>
						</optgroup>
					</select>

					<button type="button" class="button add-field">
						<?php esc_html_e( 'Add Field', 'simple-pay' ); ?>
					</button>
				</div>
			</td>
		</tr>
		<tr class="simpay-panel-field">
			<td style="padding-bottom: 0; border-bottom: 0;">
				<div id="simpay-custom-fields-wrap" class="panel simpay-metaboxes-wrapper">
					<div class="simpay-custom-fields simpay-metaboxes ui-sortable">
						<?php
						foreach ( $fields as $k => $field ) :
							// Don't render settings for custom field types that don't exist,
							// possibly from an upgrade or downgrade.
							if ( ! isset( $field_types[ $field['type'] ] ) ) :
								continue;
							endif;

							$counter = $k + 1;

							// Label compat based on static custom fields.
							$label = isset( $field['label'] )
								? $field['label']
								: '';

							switch ( $field['type'] ) {
								case 'custom_amount':
									$label = simpay_get_saved_meta( $post_id, '_custom_amount_label' );
									break;
								case 'plan_select':
									$label = simpay_get_saved_meta( $post_id, '_plan_select_form_field_label' );
									break;
							}

							$field['label'] = $label;

							echo get_custom_field( $field['type'], $counter, $field ); // WPCS: XSS okay.
						endforeach;
						?>
					</div>
				</div>
			</td>
		</tr>
	</tbody>
</table>

	<?php
	/** This filter is documented in includes/core/post-types/simple-pay/edit-form-custom-fields.php */
	do_action( 'simpay_admin_after_custom_fields' );

	echo simpay_docs_link( __( 'Help docs for Custom Form Fields', 'simple-pay' ), 'custom-form-fields', 'form-settings' );

	/**
	 * Allows further output after "Custom Fields" Payment Form
	 * settings tab content.
	 *
	 * @since 3.0.0
	 */
	do_action( 'simpay_custom_field_panel' );
}
// Remove Lite "Custom Fields" tab content.
remove_action(
	'simpay_form_settings_meta_form_display_panel',
	'SimplePay\\Core\\Post_Types\\Simple_Pay\\Edit_Form\\add_custom_fields'
);
add_action( 'simpay_form_settings_meta_form_display_panel', __NAMESPACE__ . '\\add_custom_fields' );

/**
 * Retrieves the markup for a custom field.
 *
 * @since 3.8.0
 *
 * @param int   $type    Custom field type.
 * @param int   $counter Custom field counter.
 * @param array $field   Custom field arguments.
 * @return string Custom field markup.
 */
function get_custom_field( $type, $counter, $field = array() ) {
	$field_types = get_custom_field_types();

	// Generate a label.
	$accordion_label = '';

	if ( isset( $field['label'] ) && ! empty( $field['label'] ) ) {
		$accordion_label = $field['label'];
	} elseif ( isset( $field['placeholder'] ) && ! empty( $field['placeholder'] ) ) {
		$accordion_label = $field['placeholder'];
	} else {
		$accordion_label = $field_types[ $type ]['label'];
	}

	$accordion_label = esc_html( $accordion_label );

	// Find the template.
	$admin_field_template = SIMPLE_PAY_INC . 'pro/post-types/simple-pay/edit-form-custom-fields/custom-fields-' . simpay_dashify( $type ) . '-html.php';

	/**
	 * Filters the template for outputting a Payment Form's custom field.
	 *
	 * @since 3.0.0
	 *
	 * @param string $admin_field_template Field path.
	 */
	$admin_field_template = apply_filters( 'simpay_admin_' . esc_attr( $type ) . '_field_template', $admin_field_template );

	$uid = isset( $field['uid'] ) ? $field['uid'] : $counter;

	ob_start();
	?>

<div
	id="simpay-custom-field-<?php echo esc_attr( simpay_dashify( $type ) . $counter ); ?>-postbox"
	class="postbox closed simpay-field-metabox simpay-metabox simpay-custom-field-<?php echo simpay_dashify( $type ); ?>"
	data-type="<?php echo esc_attr( $type ); ?>"
	aria-expanded="false"
>
	<button type="button" class="simpay-handlediv">
		<span class="screen-reader-text">
			<?php echo esc_html( sprintf( __( 'Toggle custom field: %s', 'simple-pay' ), $accordion_label ) ); ?>
		</span>
		<span class="toggle-indicator" aria-hidden="true"></span>
	</button>

	<h2 class="simpay-hndle ui-sortable-handle">
		<span class="custom-field-dashicon dashicons dashicons-menu"></span>

		<strong>
			<?php echo esc_html( $accordion_label ); ?>
			</strong>

		<div class="simpay-field-type"><?php echo esc_html( $field_types[ $type ]['label'] ); ?></div>
	</h2>

	<div class="simpay-field-data simpay-metabox-content inside">
		<table>
			<?php
			if ( file_exists( $admin_field_template ) ) :
				simpay_print_field(
					array(
						'type'    => 'standard',
						'subtype' => 'hidden',
						'name'    => '_simpay_custom_field[' . $type . '][' . $counter . '][id]',
						'id'      => 'simpay-' . $type . '-' . $counter . '-id',
						'value'   => ! empty( $field['id'] ) ? $field['id'] : $uid,
					)
				);

				simpay_print_field(
					array(
						'type'    => 'standard',
						'subtype' => 'hidden',
						'id'      => 'simpay-' . $type . '-' . $counter . '-uid',
						'class'   => array( 'field-uid' ),
						'name'    => '_simpay_custom_field[' . $type . '][' . $counter . '][uid]',
						'value'   => $uid,
					)
				);

				simpay_print_field(
					array(
						'type'    => 'standard',
						'subtype' => 'hidden',
						'id'      => 'simpay-' . $type . '-' . $counter . '-order',
						'class'   => array( 'field-order' ),
						'name'    => '_simpay_custom_field[' . $type . '][' . $counter . '][order]',
						'value'   => isset( $field['order'] ) ? $field['order'] : $counter,
					)
				);

				include $admin_field_template;

				/**
				 * Allows further output after a specific custom field type.
				 *
				 * @since 3.0.0
				 */
				do_action( 'simpay_after_' . $type . '_meta' );
			endif;
			?>
		</table>

		<div class="simpay-metabox-content-actions">
			<button type="button" class="button-link simpay-remove-field-link">
				<?php esc_html_e( 'Remove', 'simple-pay' ); ?>
			</button>

			<div class="simpay-metabox-content-actions__field-id">
				<label for="<?php echo esc_attr( 'simpay-' . $type . '-' . $counter . '-' . $uid ); ?>">
					<?php esc_html_e( 'Field ID', 'simple-pay' ); ?>:
				</label>

				<input type="text" value="<?php echo absint( $uid ); ?>" id="<?php echo esc_attr( 'simpay-' . $type . '-' . $counter . '-' . $uid ); ?>" readonly />

				<a href="<?php echo esc_url( simpay_docs_link( '', 'custom-form-fields#field-id', 'global-settings', true ) ); ?>" class="simpay-docs-icon" target="_blank" rel="noopener noreferrer">
					<span class="dashicons dashicons-editor-help"></span>
					<span class="screen-reader-text"><?php esc_html_e( 'Find out more about the field ID.', 'simple-pay' ); ?></span>
				</a>
			</div>
		</div>
	</div>
</div>

	<?php
		return ob_get_clean();
}

/**
 * Ensures Payment Forms have required fields and remove unnecessary fields.
 *
 * @since 3.8.0
 *
 * @param array  $fields            Payment Form custom fields.
 * @param int    $form_id           Payment Form ID.
 * @param string $form_display_type Payment Form display type.
 * @return array
 */
function add_missing_custom_fields( $fields, $form_id, $form_display_type = 'embedded' ) {
	$amount_type = isset( $_POST['_amount_type'] )
		? sanitize_text_field( $_POST['_amount_type'] )
		: '';

	$count = count( $fields );

	// Form display type-specific.
	switch ( $form_display_type ) {
		case 'embedded':
		case 'overlay':
			// Ensure "Customer Email" exists.
			if ( ! isset( $fields['email'] ) ) {
				$fields['email'][] = array(
					'uid'   => $count,
					'id'    => 'simpay_' . $form_id . '_email',
					'label' => 'Email Address',
				);

				$count++;
			}

			// Ensure "Payment Methods" exist.
			if ( ! isset( $fields['card'] ) ) {
				$fields['card'][] = array(
					'order' => 9998,
					'uid'   => $count,
					'id'    => 'simpay_' . $form_id . '_card',
				);

				$count++;
			}

			// Ensure "Checkout" button exists.
			if ( ! isset( $fields['checkout_button'] ) ) {
				$fields['checkout_button'][] = array(
					'order' => 9999,
					'uid'   => $count,
					'id'    => 'simpay_' . $form_id . '_checkout_button',
				);

				$count++;
			}

			// Ensure "Payment Button" exists.
			if ( 'overlay' === $form_display_type ) {
				if ( ! isset( $fields['payment_button'] ) ) {
					$fields['payment_button'][] = array(
						'uid' => $count,
						'id'  => 'simpay_' . $form_id . '_payment_button',
					);

					$count++;
				}
				// Remove "Payment Button".
			} else {
				unset( $fields['payment_button'] );
			}

			break;
		default:
			// Ensure "Payment Button" exists.
			if ( ! isset( $fields['payment_button'] ) ) {
				$fields['payment_button'][] = array(
					'order' => 9999,
					'uid'   => $count,
					'id'    => 'simpay_' . $form_id . '_payment_button',
				);

				$count++;
			}

			// Remove unnecessary fields.
			unset( $fields['card'] );
			unset( $fields['checkout_button'] );
	}

	// Ensure "Subscription Plan Selector" exists if using Subscription Plans.
	$sub_type = isset( $_POST['_subscription_type'] )
		? sanitize_text_field( $_POST['_subscription_type'] )
		: '';

	if ( 'disabled' !== $sub_type ) {
		if ( ! isset( $fields['plan_select'] ) ) {
			$fields['plan_select'][] = array(
				'uid' => $count,
				'id'  => 'simpay_' . $form_id . '_plan_select_' . $count,
			);

			update_post_meta( $form_id, '_plan_select_form_field_label', 'Choose a Plan' );
		}

		unset( $fields['recurring_amount_toggle'] );
		unset( $fields['custom_amount'] );

		$count++;
	} elseif ( 'disabled' === $sub_type ) {
		unset( $fields['plan_select'] );
	}

	// Ensure "Custom Amount" field exists if using a custom amount, and no Subscription.
	if (
		'one_time_custom' === $amount_type &&
		! isset( $fields['custom_amount'] ) &&

		// Ensure we aren't using a previously saved value, and Subscriptions
		// are actually enabled.
		'disabled' === $sub_type
	) {
		$fields['custom_amount'][] = array(
			'uid'   => $count,
			'id'    => 'simpay_' . $form_id . '_custom_amount_' . $count,
			'label' => 'Custom Amount',
		);

		$count++;
	} elseif ( 'one_time_custom' !== $amount_type || 'disabled' !== $sub_type ) {
		unset( $fields['custom_amount'] );
	}

	// General sorting template for auto-added fields.
	$fields = array_merge(
		array_flip(
			array(
				'email',
				'plan_select',
				'custom_amount',
				'card',
				'checkout_button',
			)
		),
		$fields
	);

	// "Payment Button" should always be last.
	if ( isset( $fields['payment_button'] ) ) {
		$payment_button = $fields['payment_button'];
		unset( $fields['payment_button'] );

		$fields['payment_button'] = $payment_button;
	}

	// Remove empty/invalid fields after sorting.
	$fields = array_filter(
		$fields,
		function( $field ) {
			return true === is_array( $field );
		}
	);

	return $fields;
}
