<?php
/**
 * Simple Pay: Edit form payment options
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds "Subscriptions are enabled" notice if Subscriptions are in use.
 *
 * @since 3.8.0
 *
 * @param int $post_id Current Payment Form ID.
 */
function maybe_hide_amount_options( $post_id ) {
	if ( false === simpay_subscriptions_enabled() ) {
		return;
	}
	?>

<table class="simpay-show-if" data-if="_subscription_type" data-is="user single">
	<tr class="simpay-panel-field">
		<td style="padding-top: 14px;">
			<div class="notice inline notice-info">
			<?php
			echo wpautop(
				wp_kses(
					sprintf(
						__( 'Subscriptions are enabled. <a href="#" class="simpay-tab-link" data-show-tab="%1$s">See Subscription Options</a> to set a recurring payment amount.', 'simple-pay' ),
						'simpay-subscription_options'
					),
					array(
						'a' => array(
							'href'          => array(),
							'class'         => array(),
							'data-show-tab' => array(),
						),
					)
				)
			);
			?>
			</div>
		</td>
	</tr>
</table>

	<?php
}
add_action( 'simpay_form_settings_meta_payment_options_panel', __NAMESPACE__ . '\\maybe_hide_amount_options', 9 );

/**
 * Adds "Amount Type" setting to "Payment Options" Payment Form
 * settings tab.
 *
 * @since 3.8.0
 *
 * @param int $post_id Current Payment Form ID.
 */
function add_payment_options( $post_id ) {
	?>

<tr class="simpay-panel-field simpay-show-if" data-if="_subscription_type" data-is="disabled">
	<th>
		<label for="_amount"><?php esc_html_e( 'Amount Type', 'simple-pay' ); ?></label>
	</th>
	<td>
		<?php
		$amount_type = simpay_get_saved_meta( $post_id, '_amount_type', 'one_time_set' );

		simpay_print_field(
			array(
				'type'    => 'radio',
				'name'    => '_amount_type',
				'id'      => '_amount_type',
				'value'   => $amount_type,
				'class'   => array(
					'simpay-field-text',
					'simpay-multi-toggle',
				),
				'options' => array(
					'one_time_set'    => esc_html__( 'One-Time Set Amount', 'simple-pay' ),
					'one_time_custom' => esc_html__( 'One-Time Custom Amount', 'simple-pay' ),
				),
				'inline'  => 'inline',
			)
		);
		?>

		<p class="description">
			<?php
			if ( false === simpay_subscriptions_enabled() ) :
				printf(
					wp_kses(
						__( '<a href="%s" target="_blank" rel="noopener noreferrer">Upgrade your license</a> to connect Stripe subscriptions to your payment forms.', 'simple-pay' ),
						array(
							'a' => array(
								'href'   => array(),
								'target' => array(),
							),
						)
					),
					simpay_my_account_url( 'form-settings' )
				);
			endif;
			?>
		</p>
	</td>
</tr>

	<?php
}
add_action( 'simpay_amount_options', __NAMESPACE__ . '\\add_payment_options' );

/**
 * Adds "Custom Amount" setting to "Payment Options" Payment Form
 * settings tab.
 *
 * @since 3.8.0
 *
 * @param int $post_id Current Payment Form ID.
 */
function add_custom_amount_options( $post_id ) {
	$currency_position = simpay_get_currency_position();
	?>

<div class="simpay-show-if" data-if="_subscription_type" data-is="disabled">
	<table class="simpay-show-if" data-if="_amount_type" data-is="one_time_custom">
		<tbody>
			<tr class="simpay-panel-field">
				<th>
					<label for="_minimum_amount">
						<?php esc_html_e( 'Minimum Custom Amount', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<div class="simpay-currency-field">
						<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-left"><?php echo simpay_get_saved_currency_symbol(); ?></span>
						<?php endif; ?>

						<?php
						$minimum_amount = simpay_get_saved_meta( $post_id, '_minimum_amount', simpay_global_minimum_amount() );

						simpay_print_field(
							array(
								'type'        => 'standard',
								'subtype'     => 'tel',
								'name'        => '_minimum_amount',
								'id'          => '_minimum_amount',
								'value'       => $minimum_amount,
								'class'       => array(
									'simpay-field-tiny',
									'simpay-amount-input',
									'simpay-minimum-amount-required',
								),
								'placeholder' => simpay_format_currency( simpay_global_minimum_amount(), simpay_get_setting( 'currency' ), false ),
							)
						);
						?>

						<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-right"><?php echo simpay_get_saved_currency_symbol(); ?></span>
						<?php endif ?>
					</div>
				</td>
			</tr>

			<tr class="simpay-panel-field">
				<th>
					<label for="_custom_amount_default">
						<?php esc_html_e( 'Default Custom Amount', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<div class="simpay-currency-field">
						<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-left"><?php echo simpay_get_saved_currency_symbol(); ?></span>
						<?php endif; ?>

						<?php
						$custom_amount_default = simpay_get_saved_meta( $post_id, '_custom_amount_default', '' );
						simpay_print_field(
							array(
								'type'    => 'standard',
								'subtype' => 'tel',
								'name'    => '_custom_amount_default',
								'id'      => '_custom_amount_default',
								'value'   => $custom_amount_default,
								'class'   => array(
									'simpay-field-tiny',
									'simpay-amount-input',
									'simpay-allow-blank-amount',
									'simpay-minimum-amount-required',
								),
							)
						);
						?>

						<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-right"><?php echo simpay_get_saved_currency_symbol(); ?></span>
						<?php endif; ?>
					</div>

					<p class="description">
						<?php esc_html_e( 'The custom amount field will load with this amount set by default.', 'simple-pay' ); ?>
					</p>
				</td>
			</tr>
		</tbody>
	</table>
</div>
	<?php
}
add_action( 'simpay_admin_after_amount_options', __NAMESPACE__ . '\\add_custom_amount_options' );
