<?php
/**
 * Simple Pay: Edit form Stripe Checkout
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form;

use SimplePay\Pro\Payment_Methods;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Hides the "Stripe Checkout Display" tab by default.
 *
 * Tab will become visible if the Form Display Type is "Stripe Checkout".
 *
 * @since 3.8.0
 *
 * @param array $tabs Payment Form settings tabs.
 * @return array $tabs Payment Form settings tabs.
 */
function hide_tab( $tabs ) {
	$tabs['stripe_checkout']['class'] = array(
		'simpay-show-if',
	);

	return $tabs;
}
add_filter( 'simpay_form_settings_meta_tabs_li', __NAMESPACE__ . '\\hide_tab' );

// Remove "Company Name" and "Item Description". It will be relocated.
//
// @see includes/pro/post-types/simple-pay/edit-form-display.php
remove_action(
	'simpay_admin_before_stripe_checkout_rows',
	'SimplePay\\Core\\Post_Types\\Simple_Pay\\Edit_Form\\add_company_info',
	20
);

/**
 * Adds Payment Method settings.
 *
 * @global WP_Post $post
 *
 * @since 3.8.0
 */
function add_payment_methods() {
	global $post;

	// Retrieve saved Payment Methods for the given context.
	$context         = 'stripe-checkout';
	$payment_methods = simpay_get_saved_meta( $post->ID, '_payment_methods', array() );

	if ( empty( $payment_methods ) ) {
		$payment_methods = array(
			array(
				'id' => 'card',
			),
		);
	} else {
		$payment_methods = isset( $payment_methods[ $context ] )
			? $payment_methods[ $context ]
			: array();
	}

	// Retrieve available Payment Methods for the given context.
	$available_payment_methods   = Payment_Methods\get_stripe_checkout_payment_methods();
	$payment_methods_with_config = array();

	// Load settings.
	include SIMPLE_PAY_INC . 'pro/admin/metaboxes/views/partials/payment-methods.php';
}
add_action( 'simpay_admin_before_stripe_checkout_rows', __NAMESPACE__ . '\\add_payment_methods' );

/**
 * Allow custom form styles to be used on the frontend.
 *
 * @since 3.8.0
 *
 * @param int $form_id Current Payment Form ID.
 */
function add_form_styles_setting( $post_id ) {
	?>

<tr class="simpay-panel-field">
	<th>
		<label for="_enable_stripe_checkout_form_styles">
			<?php esc_html_e( 'Enable Form Styles', 'simple-pay' ); ?>
		</label>
	</th>
	<td>
		<?php
		simpay_print_field(
			array(
				'type'        => 'checkbox',
				'name'        => '_enable_stripe_checkout_form_styles',
				'id'          => '_enable_stripe_checkout_form_styles',
				'value'       => simpay_get_saved_meta( $post_id, '_enable_stripe_checkout_form_styles', 'no' ),
				'description' => esc_html__( 'Apply plugin styling to form fields that appear on-page. Otherwise the styles will inherit from the current theme.', 'simple-pay' ),
			)
		);
		?>
	</td>
</tr>

	<?php
}
add_action( 'simpay_after_checkout_button_text', __NAMESPACE__ . '\\add_form_styles_setting' );

/**
 * Output a link back to "Custom Form Fields" under the "Checkout Button Text" field.
 *
 * @since 3.8.0
 */
function add_custom_form_fields_link() {
	$message = sprintf(
		__( 'Configure the on-site Payment Button in the %1$sCustom Form Fields%2$s settings.', 'simple-pay' ),
		'<a href="#" class="simpay-tab-link" data-show-tab="simpay-form_display">',
		'</a>'
	);
	?>

<tr class="simpay-panel-field">
	<td>
		<div class="notice inline notice-info" style="margin-top: 18px;">
			<p>
				<?php echo wp_kses_post( $message ); ?>
			</p>
		</div>
	</td>
</tr>

	<?php
}
remove_action(
	'simpay_admin_before_stripe_checkout_rows',
	'SimplePay\\Core\\Post_Types\\Simple_Pay\\Edit_Form\\add_custom_form_fields_link'
);
add_action( 'simpay_admin_before_stripe_checkout_rows', __NAMESPACE__ . '\\add_custom_form_fields_link' );
