<?php
/**
 * Simple Pay: Edit form Subscription options
 *
 * @package SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.8.0
 */

namespace SimplePay\Pro\Post_Types\Simple_Pay\Edit_Form;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds hidden fields for Subscription settings that control
 * the visibility of other settings when Subscriptions are not
 * available on the current install.
 *
 * @since 3.8.1
 */
function add_hidden_subscription_options() {
	echo '<div style="display: none;">';
	echo '<input type="radio" name="_subscription_type" value="disabled" checked />';
	echo '<input type="radio" name="_subscription_custom_amount" value="disabled" checked />';
	echo '</div>';
}
// Remove Lite tab content if Subscriptions are enabled.
if ( true === simpay_subscriptions_enabled() ) {
	remove_action(
		'simpay_form_settings_meta_subscription_display_panel',
		'SimplePay\\Core\\Post_Types\\Simple_Pay\\Edit_Form\\add_subscription_options'
	);
	// Or add hidden fields for toggles.
} else {
	add_action( 'simpay_form_settings_meta_subscription_display_panel', __NAMESPACE__ . '\\add_hidden_subscription_options' );
}

/**
 * Adds "Subscription Options" Payment Form settings tab content.
 *
 * @since 3.8.0
 *
 * @param int $post_id Current Payment Form ID.
 */
function add_subscription_options( $post_id ) {
	$currency_position = simpay_get_currency_position();

	/** This filter is documented in includes/core/shortcodes.php */
	$form = apply_filters( 'simpay_form_view', '', $post_id );

	if ( empty( $form ) ) {
		$form = new Default_Form( $post_id );
	}

	$active_plans_list = simpay_get_plan_list( $form );
	$plan_options      = array_merge(
		array(
			'empty' => esc_html__( 'Choose a plan&hellip;', 'simple-pay' ),
		),
		$active_plans_list
	);

	// Retrieve saved Plans and filter Plans that no longer exist.
	$plans = $form->get_subscription_plans();
	$plans = array_filter(
		$plans,
		function( $plan ) use ( $active_plans_list ) {
			return isset( $active_plans_list[ $plan['select_plan'] ] );
		}
	);

	$subscription_type = simpay_get_saved_meta( $post_id, '_subscription_type', 'disabled' );
	?>

<table>
	<tbody class="simpay-panel-section">
		<tr class="simpay-panel-field">
			<th>
				<label for="_subscription_type">
					<?php esc_html_e( 'Subscription Type', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<?php
				simpay_print_field(
					array(
						'type'       => 'radio',
						'name'       => '_subscription_type',
						'id'         => '_subscription_type',
						'value'      => $subscription_type,
						'class'      => array(
							'simpay-field-text',
							'simpay-multi-toggle',
							'simpay-disable-amount',
						),
						'options'    => array(
							'single'   => esc_html__( 'Single Plan', 'simple-pay' ),
							'user'     => esc_html__( 'User Selected Plan', 'simple-pay' ),
							'disabled' => esc_html__( 'Disabled', 'simple-pay' ),
						),
						'inline'     => 'inline',
						'attributes' => array(
							'data-disable-amount-check' => 'disabled',
						),
					)
				);
				?>
			</td>
		</tr>

		<tr class="simpay-panel-field simpay-show-if" data-if="_subscription_type" data-is="user single">
			<td style="padding-top: 0; padding-bottom: 0; border-bottom: 0;">
				<div class="notice inline notice-info is-dismissable" style="margin-top: 18px;">
					<p>
						<?php
						echo wp_kses(
							sprintf(
								/** translators: %1$s: Current payment mode. %2$s Link to Plans in Stripe Dashboard, do not translate. %3$s Link to documentation, do not translate. %4$s Closing anchor tag, do not translate. */
								__( 'While in %1$s you may only select Plans from your %2$s%1$s Products%3$s. Products are configured separately for test mode and live mode &mdash; Products and Plans that you add in one mode are not available in the other. Be sure to configure your Products and Plans for both modes and review this form\'s settings when toggling between modes.', 'simple-pay' ),
								'<strong>' . ( $form->is_livemode() ? __( 'Live Mode', 'simple-pay' ) : __( 'Test Mode', 'simple-pay' ) ) . '</strong>',
								'<a href="https://dashboard.stripe.com/' . ( $form->is_livemode() ? '' : 'test/' ) . 'plans" target="_blank" rel="noopener noreferrer">',
								'</a>'
							),
							array(
								'a'      => array(
									'href'   => array(),
									'target' => array(),
									'rel'    => array(),
								),
								'strong' => array(),
							)
						);
						?>
					</p>
					<p>
						<?php
						echo wp_kses(
							sprintf(
								/** translators: %1$s: Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
								__( 'For more information please see the %1$sdocumentation%2$s.', 'simple-pay' ),
								'<a href="' . simpay_docs_link( '', 'adding-subscription-plans-stripe', 'form-settings', true ) . '" target="_blank" rel="noopener noreferrer">',
								'</a>'
							),
							array(
								'a' => array(
									'href'   => array(),
									'target' => array(),
									'rel'    => array(),
								),
							)
						);
						?>
					</p>
				</div>
			</td>
		</tr>
	</tbody>
</table>

<table class="simpay-show-if" data-if="_subscription_type" data-is="user single">
	<tr class="simpay-panel-field">
		<th>
			<label for="_subscription_custom_amount">
				<?php esc_html_e( 'Custom Amount', 'simple-pay' ); ?>
			</label>
		</th>
		<td>
			<?php
			$multi_plan_custom_amount = simpay_get_saved_meta( $post_id, '_subscription_custom_amount', 'disabled' );

			simpay_print_field(
				array(
					'type'    => 'radio',
					'name'    => '_subscription_custom_amount',
					'id'      => '_subscription_custom_amount',
					'value'   => $multi_plan_custom_amount,
					'class'   => array(
						'simpay-field-text',
						'simpay-multi-toggle',
					),
					'options' => array(
						'enabled'  => esc_html__( 'Enabled', 'simple-pay' ),
						'disabled' => esc_html__( 'Disabled', 'simple-pay' ),
					),
					'inline'  => 'inline',
				)
			);
			?>
		</td>
	</tr>
</table>

<table>
	<tbody class="simpay-show-if" data-if="_subscription_custom_amount" data-is="disabled">
		<tr class="simpay-panel-field simpay-show-if" data-if="_subscription_type" data-is="single">
			<th>
				<label for="_single_plan">
					<?php esc_html_e( 'Subscription Plan', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<?php
				$attr          = array();
				$custom_amount = simpay_get_saved_meta( $post_id, '_subscription_custom_amount' );

				if ( 'enabled' === $custom_amount ) :
					$attr['disabled'] = 'disabled';
				endif;

				// Try livemode toggle.
				$single_plan  = simpay_get_saved_meta(
					$form->id,
					'_single_plan' . ( false === $form->is_livemode() ? '_test' : '' )
				);

				// Fall back to current.
				if ( empty( $single_plan ) ) {
					$single_plan = simpay_get_saved_meta( $form->id, '_single_plan' );
				}

				simpay_print_field(
					array(
						'type'       => 'select',
						'name'       => '_single_plan',
						'id'         => '_single_plan',
						'value'      => $single_plan,
						'class'      => array(
							'simpay-field-text',
						),
						'options'    => $plan_options,
						'attributes' => $attr,
					)
				);
				?>
			</td>
		</tr>
	</tbody>
</table>

<table id="simpay-multi-plans" class="simpay-multi-subs-wrap simpay-show-if" data-if="_subscription_type" data-is="user">
	<tr class="simpay-panel-field">
		<th>
			<label for="simpay-user-sub-add-plan">
				<?php esc_html_e( 'Subscription Plans', 'simple-pay' ); ?>
			</label>
		</th>

		<td style="border-bottom: 0; padding-bottom: 0;">
			<div class="toolbar toolbar-top">
				<select name="simpay-user-sub-add-plan" id="simpay-user-sub-add-plan" class="simpay-field-select">
				<?php foreach ( $plan_options as $id => $value ) : ?>
					<option value="<?php echo esc_attr( $id ); ?>"><?php echo esc_html( $value ); ?></option>
				<?php endforeach; ?>
				</select>

				<input type="hidden" id="simpay_add_plan_nonce" value="<?php echo esc_attr( wp_create_nonce( 'simpay_add_plan_nonce' ) ); ?>" />
				<button class="simpay-add-plan button button-secondary"><?php esc_html_e( 'Add Plan', 'simple-pay' ); ?></button>
			</div>
		</td>
	</tr>

	<tr class="simpay-panel-field">
		<td class="simpay-user-subs <?php echo esc_attr( empty( $plans ) ? 'is-empty' : '' ); ?>">
			<div id="simpay-user-sub-wrap" class="panel simpay-metaboxes-wrapper">
				<div class="simpay-metaboxes ui-sortable <?php echo esc_attr( empty( $plans ) ? 'is-empty' : '' ); ?>">
					<?php
					if ( ! empty( $plans ) ) :
						foreach ( $plans as $plan_counter => $plan ) :
							echo get_subscription_plan( $plan, $plan_counter, $post_id ); // WPCS: XSS okay.
						endforeach;
					endif;
					?>
				</div>
			</div>
		</td>
	</tr>
	<tr class="simpay-panel-field">
		<th>
			<label for="_multi_plan_display">
				<?php esc_html_e( 'Display Style', 'simple-pay' ); ?>
			</label>
		</th>
		<td>
			<?php
			simpay_print_field(
				array(
					'type'    => 'radio',
					'name'    => '_multi_plan_display',
					'id'      => '_multi_plan_display',
					'value'   => simpay_get_saved_meta( $post_id, '_multi_plan_display', 'radio' ),
					'class'   => array(
						'simpay-field-text',
					),
					'options' => array(
						'radio'    => esc_html__( 'Radio', 'simple-pay' ),
						'dropdown' => esc_html__( 'Dropdown', 'simple-pay' ),
					),
					'inline'  => 'inline',
				)
			);
			?>
		</td>
	</tr>
</table>

<table class="simpay-inner-table simpay-show-if" data-if="_subscription_type" data-is="user single">
	<tbody>
		<tr class="simpay-panel-field">
			<th>
				<label for="_setup_fee">
					<?php esc_html_e( 'Initial Setup Fee', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<div class="simpay-currency-field">
					<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-left">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif; ?>

					<?php
					$setup_fee = simpay_get_saved_meta( $post_id, '_setup_fee', '0' );

					simpay_print_field(
						array(
							'type'    => 'standard',
							'subtype' => 'tel',
							'name'    => '_setup_fee',
							'id'      => '_setup_fee',
							'value'   => $setup_fee,
							'class'   => array(
								'simpay-field-tiny',
								'simpay-amount-input',
								'simpay-allow-blank-amount',
								'simpay-minimum-amount-required',
							),
						)
					);
					?>

					<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-right">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif; ?>
				</div>

				<p class="description">
					<?php esc_html_e( 'An optional amount to add to the first payment only.', 'simple-pay' ); ?>

					<span class="simpay-show-if" data-if="_subscription_type" data-is="user">
						<?php echo ' ' . esc_html__( 'Added on top of existing Plan-level setup fees.', 'simple-pay' ); ?>
					</span>
				</p>
			</td>
		</tr>
	</tbody>

	<tbody class="simpay-show-if" data-if="_subscription_custom_amount" data-is="enabled">
		<tr class="simpay-panel-field">
			<th>
				<label for="_custom_plan_label">
					<?php esc_html_e( 'Custom Amount Label', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<?php
				simpay_print_field(
					array(
						'type'        => 'standard',
						'subtype'     => 'text',
						'name'        => '_custom_plan_label',
						'id'          => '_custom_plan_label',
						'value'       => simpay_get_saved_meta( $post_id, '_custom_plan_label' ),
						'class'       => array(
							'simpay-field-text',
						),
						'placeholder' => esc_attr__( 'Other amount', 'simple-pay' ),
					)
				);
				?>
			</td>
		</tr>

		<tr class="simpay-panel-field">
			<th>
				<label for="_multi_plan_minimum_amount">
					<?php esc_html_e( 'Minimum Amount', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<div class="simpay-currency-field">
					<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-left">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif ?>

					<?php
					// Check saved currency and set default to 100 or 1 accordingly and set steps and class.
					$multi_plan_minimum_amount = simpay_get_saved_meta( $post_id, '_multi_plan_minimum_amount', simpay_format_currency( simpay_global_minimum_amount(), simpay_get_setting( 'currency' ), false ) );

					simpay_print_field(
						array(
							'type'        => 'standard',
							'subtype'     => 'tel',
							'name'        => '_multi_plan_minimum_amount',
							'id'          => '_multi_plan_minimum_amount',
							'value'       => $multi_plan_minimum_amount,
							'class'       => array(
								'simpay-field-tiny',
								'simpay-amount-input',
								'simpay-minimum-amount-required',
							),
							'placeholder' => simpay_format_currency( simpay_global_minimum_amount(), simpay_get_setting( 'currency' ), false ),
						)
					);
					?>

					<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-right">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif; ?>
				</div>
			</td>
		</tr>

		<tr class="simpay-panel-field simpay-show-if" data-if="_subscription_type" data-is="single">
			<th>
				<label for="_multi_plan_default_amount">
					<?php esc_html_e( 'Default Amount', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<div class="simpay-currency-field">
					<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-left">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif; ?>

					<?php
					// Check saved currency and set default to 100 or 1 accordingly and set steps and class.
					$multi_plan_default_amount = simpay_get_saved_meta( $post_id, '_multi_plan_default_amount', '' );

					simpay_print_field(
						array(
							'type'    => 'standard',
							'subtype' => 'tel',
							'name'    => '_multi_plan_default_amount',
							'id'      => '_multi_plan_default_amount',
							'value'   => $multi_plan_default_amount,
							'class'   => array(
								'simpay-field-tiny',
								'simpay-amount-input',
								'simpay-allow-blank-amount',
								'simpay-minimum-amount-required',
							),
						)
					);
					?>

					<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
						<span class="simpay-currency-symbol simpay-currency-symbol-right">
							<?php echo simpay_get_saved_currency_symbol(); ?>
						</span>
					<?php endif; ?>
				</div>

				<p class="description">
					<?php esc_html_e( 'The custom amount field will load with this amount set by default.', 'simple-pay' ); ?>
				</p>
			</td>
		</tr>

		<tr class="simpay-panel-field">
			<th>
				<label for="_plan_interval">
					<?php esc_html_e( 'Interval/Frequency', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<?php
				simpay_print_field(
					array(
						'type'       => 'standard',
						'subtype'    => 'number',
						'name'       => '_plan_interval',
						'id'         => '_plan_interval',
						'value'      => simpay_get_saved_meta( $post_id, '_plan_interval', 1 ),
						'class'      => array(
							'small-text',
						),
						'attributes' => array(
							'min' => 1,
						),
					)
				);

				simpay_print_field(
					array(
						'type'    => 'select',
						'name'    => '_plan_frequency',
						'id'      => '_plan_frequency',
						'value'   => simpay_get_saved_meta( $post_id, '_plan_frequency', 'empty' ),
						'class'   => array(
							'simpay-plan-frequency',
						),
						'options' => array(
							'month' => esc_html__( 'Month(s)', 'simple-pay' ),
							'week'  => esc_html__( 'Week(s)', 'simple-pay' ),
							'day'   => esc_html__( 'Day(s)', 'simple-pay' ),
							'year'  => esc_html__( 'Year(s)', 'simple-pay' ),
						),
					)
				);
				?>
			</td>
		</tr>
	</tbody>
	<tbody id="simpay-subscription-charge-limit">
		<tr class="simpay-panel-field">
			<th>
				<label for="_max_charges">
					<?php esc_html_e( 'Charge Limit', 'simple-pay' ); ?>
				</label>
			</th>
			<td>
				<?php
				$max_charges = simpay_get_saved_meta( $post_id, '_max_charges', '0' );

				simpay_print_field(
					array(
						'type'        => 'standard',
						'subtype'     => 'number',
						'name'        => '_max_charges',
						'id'          => '_max_charges',
						'value'       => absint( $max_charges ),
						'class'       => array(
							'small-text',
						),
						'attributes'  => array(
							'min'  => 0,
							'step' => 1,
						),
						'description' => esc_html__( 'The number of times the custom Subscription will be billed before being cancelled. Leave blank for indefinite.', 'simple-pay' ),
					)
				);
				?>

				<div class="notice inline notice-info">
					<?php echo wpautop( simpay_webhook_help_text() ); // WPCS: XSS okay ?>
				</div>
			</td>
		</tr>
	</tbody>
</table>

	<?php
	echo simpay_docs_link( __( 'Help docs for Subscription Options', 'simple-pay' ), 'subscription-options', 'form-settings' );
}
if ( true === simpay_subscriptions_enabled() ) {
	add_action( 'simpay_form_settings_meta_subscription_display_panel', __NAMESPACE__ . '\\add_subscription_options' );
}

/**
 * Retrieves the markup for a Subscription Plan.
 *
 * @since 3.8.0
 *
 * @param array $plan         Subscription Plan settings.
 * @param int   $plan_counter Current Plan count.
 * @param int   $post_id      Current Payment Form ID.
 * @return string
 */
function get_subscription_plan( $plan, $plan_counter, $post_id ) {
	/** This filter is documented in includes/core/shortcodes.php */
	$form = apply_filters( 'simpay_form_view', '', $post_id );

	if ( empty( $form ) ) {
		$form = new Default_Form( $post_id );
	}

	$active_plans_list = simpay_get_plan_list( $form );
	$currency_position = simpay_get_currency_position();
	$default           = simpay_get_saved_meta( $post_id, '_multi_plan_default_value' );

	if ( ! isset( $active_plans_list[ $plan['select_plan'] ] ) ) {
		return;
	}

	// Set first added as default.
	if ( '' === $default && 0 === $plan_counter ) {
		$default = $plan['select_plan'];
	}

	ob_start();
	?>

<div
	id="simpay-subscription-multi-plan-select-<?php echo esc_attr( $plan_counter ); ?>-postbox"
	class="postbox closed simpay-field-metabox simpay-metabox"
	aria-expanded="false"
>
	<button type="button" class="simpay-handlediv">
		<span class="screen-reader-text"><?php echo esc_html( sprintf( __( 'Toggle subscription settings: %s', 'simple-pay' ), $active_plans_list[ $plan['select_plan'] ] ) ); ?></span>
		<span class="toggle-indicator" aria-hidden="true"></span>
	</button>

	<h2 class="simpay-hndle ui-sortable-handle">
		<span class="custom-field-dashicon dashicons dashicons-menu"></span>
		<strong>
			<?php echo esc_html( $active_plans_list[ $plan['select_plan'] ] ); ?>
		</strong>
	</h2>

	<div class="simpay-field-data simpay-metabox-content inside">
		<table>
			<tr class="simpay-panel-field">
				<th>
					<label for="<?php echo esc_attr( 'simpay-subscription-multi-plan-select-' . $plan_counter ); ?>">
						<?php esc_html_e( 'Plan', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<?php
					simpay_print_field(
						array(
							'type'       => 'select',
							'name'       => '_multi_plan[' . $plan_counter . '][select_plan]',
							'id'         => 'simpay-subscription-multi-plan-select-' . $plan_counter,
							'value'      => $plan['select_plan'],
							'options'    => array_merge(
								array(
									'empty' => esc_html__( 'Choose a plan&hellip;', 'simple-pay' ),
								),
								$active_plans_list
							),
							'class'      => array(
								'simpay-field',
								'simpay-field-select',
							),
							'attributes' => array(
								'data-field-key' => $plan_counter,
							),
						)
					);
					?>
				</td>
			</tr>
			<tr class="simpay-panel-field">
				<th>
					<label for="<?php echo esc_attr( 'simpay-subscription-multi-plan-custom-label-' . $plan_counter ); ?>">
						<?php esc_html_e( 'Custom Label', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<?php
					simpay_print_field(
						array(
							'type'        => 'standard',
							'subtype'     => 'text',
							'name'        => '_multi_plan[' . $plan_counter . '][custom_label]',
							'id'          => 'simpay-subscription-multi-plan-custom-label-' . $plan_counter,
							'value'       => isset( $plan['custom_label'] ) ? $plan['custom_label'] : '',
							'class'       => array(
								'simpay-field-text',
								'simpay-label-input',
							),
							'attributes'  => array(
								'data-field-key' => $plan_counter,
							),
							'description' => __( 'Displayed instead of Plan name.', 'simple-pay' ),
						)
					);
					?>
				</td>
			</tr>
			<tr class="simpay-panel-field">
				<th>
					<label for="<?php echo esc_attr( 'simpay-subscription-multi-plan-setup-fee-' . $plan_counter ); ?>">
						<?php esc_html_e( 'Setup Fee', 'simple-pay' ); ?>
					</label>
				</th>
				<td>
					<div class="simpay-currency-field">
						<?php if ( in_array( $currency_position, array( 'left', 'left_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-left"><?php echo simpay_get_saved_currency_symbol(); ?></span>
						<?php endif; ?>

						<?php
						simpay_print_field(
							array(
								'type'       => 'standard',
								'subtype'    => 'tel',
								'name'       => '_multi_plan[' . $plan_counter . '][setup_fee]',
								'id'         => 'simpay-subscription-multi-plan-setup-fee-' . $plan_counter,
								'value'      => isset( $plan['setup_fee'] ) ? $plan['setup_fee'] : '',
								'attributes' => array(
									'data-field-key' => $plan_counter,
								),
								'class'      => array(
									'simpay-field-tiny',
									'simpay-amount-input',
									'simpay-allow-blank-amount',
									'simpay-minimum-amount-required',
								),
							)
						);
						?>

						<?php if ( in_array( $currency_position, array( 'right', 'right_space' ), true ) ) : ?>
							<span class="simpay-currency-symbol simpay-currency-symbol-right">
								<?php echo simpay_get_saved_currency_symbol(); ?>
							</span>
						<?php endif; ?>
					</div>

					<p class="description">
						<?php esc_html_e( 'An additional amount to add to the first charge.', 'simple-pay' ); ?>
					</p>
				</td>
			</tr>
			<tr class="simpay-panel-field">
				<th>
					<label for="<?php echo esc_attr( 'simpay-subscription-multi-plan-max-charges-' . $plan_counter ); ?>">
						<?php esc_html_e( 'Charge Limit', 'simple-pay' ); ?>
					</label>
				</th>
				<td class="simpay-multi-plan-max-charges">
					<?php
					simpay_print_field(
						array(
							'type'        => 'standard',
							'subtype'     => 'number',
							'name'        => '_multi_plan[' . $plan_counter . '][max_charges]',
							'id'          => 'simpay-subscription-multi-plan-max-charges-' . $plan_counter,
							'value'       => isset( $plan['max_charges'] ) && ! empty( $plan['max_charges'] ) ? $plan['max_charges'] : 0,
							'class'       => array(
								'small-text',
							),
							'attributes'  => array(
								'data-field-key' => $plan_counter,
								'min'            => 0,
								'step'           => 1,
							),
							'description' => esc_html__( 'The number of times the Subscription will be billed before being cancelled. Leave blank for indefinite.', 'simple-pay' ),
						)
					);
					?>
					<div class="notice inline notice-info">
						<?php echo wpautop( simpay_webhook_help_text() ); // WPCS: XSS okay ?>
					</div>
				</td>
			</tr>
		</table>

		<div class="simpay-metabox-content-actions">
			<button class="button-link simpay-remove-plan"><?php esc_html_e( 'Remove', 'simple-pay' ); ?></button>

			<label for="<?php echo esc_attr( 'simpay-subscription-multi-plan-default-' . $plan_counter ); ?>" style="display: flex; align-items: center; padding: 8px 0;">
				<input
					type="checkbox"
					id="<?php echo esc_attr( 'simpay-subscription-multi-plan-default-' . $plan_counter ); ?>"
					name="_multi_plan_default_value"
					style="margin: 0 4px 0 0;"
					value="<?php echo esc_attr( $plan['select_plan'] ); ?>"
					<?php checked( $default, $plan['select_plan'] ); ?>
				/>
				<?php esc_html_e( 'Default selection', 'simple-pay' ); ?>
			</label>
		</div>
	</div>
</div>

	<?php
}
