<?php
/**
 * REST API: Charge Controller
 *
 * @package SimplePay\Core\REST_API\v2
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.9.0
 */

namespace SimplePay\Pro\REST_API\v2;

use SimplePay\Core\REST_API\Controller;
use SimplePay\Core\Forms\Default_Form;
use SimplePay\Core\Legacy;
use SimplePay\Pro\Payments\Charge;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Charge_Controller.
 */
class Charge_Controller extends Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpsp/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'charge';

	/**
	 * Registers the routes for Charges.
	 *
	 * @since 3.9.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			$this->rest_base . '/confirm',
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'confirm_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Allows POST requests to this endpoint with a valid nonce.
	 *
	 * @since 3.9.0
	 *
	 * @param \WP_REST_Request Request data.
	 * @return bool True with a valid nonce.
	 */
	public function create_item_permissions_check( $request ) {
		$form_values = $request['form_values'];

		if ( ! isset( $form_values['_wpnonce'] ) || ! wp_verify_nonce( $form_values['_wpnonce'], 'simpay_payment_form' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Handles an incoming request to create a Charge.
	 *
	 * @since 3.9.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type int   $customer_id Customer ID previously generated with Payment Source.
	 *   @type int   $form_id Form ID used to generate Charge data.
	 *   @type array $form_data Client-generated formData information.
	 *   @type array $form_values Values of named fields in the payment form.
	 * }
	 * @return \WP_REST_Response
	 */
	public function create_item( $request ) {
		try {
			// Gather customer information.
			$customer_id = isset( $request['customer_id'] ) ? $request['customer_id'] : false;

			if ( ! $customer_id ) {
				throw new \Exception( __( 'A customer must be provided.', 'simple-pay' ) );
			}

			// Locate form.
			if ( ! isset( $request['form_id'] ) ) {
				throw new \Exception( __( 'Unable to locate payment form.', 'simple-pay' ) );
			}

			// Gather <form> information.
			$form_id     = $request['form_id'];
			$form_data   = $request['form_data'];
			$form_values = $request['form_values'];

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			// Handle legacy form processing.
			Legacy\Hooks\simpay_process_form( $form, $form_data, $form_values, $customer_id );

			$charge_args = $this->get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id );

			/**
			 * Allows processing before a Charge is created from a Payment Form request.
			 *
			 * @since 3.9.0
			 *
			 * @param array                         $session_args Arguments used to create a PaymentIntent.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int|string                    $customer_id Stripe Customer ID, or a blank string if none is needed.
			 */
			do_action(
				'simpay_before_charge_from_payment_form_request',
				$charge_args,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			$charge = Charge\create(
				$charge_args,
				$form->get_api_request_args()
			);

			/**
			 * Allows further processing after a Charge is created from a Payment Form request.
			 *
			 * @since 3.9.0
			 *
			 * @param \Stripe\Charge                $charge Stripe Charge.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int                           $customer_id Stripe Customer ID.
			 */
			do_action(
				'simpay_after_charge_from_payment_form_request',
				$charge,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			return new \WP_REST_Response(
				array(
					'success' => true,
				)
			);
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => $e->getMessage(),
				),
				400
			);
		}
	}

	/**
	 * Uses the current payment form request to generate arguments for a Charge.
	 *
	 * @since 3.9.0
	 *
	 * @param SimplePay\Core\Abstracts\Form $form Form instance.
	 * @param array                         $form_data Form data generated by the client.
	 * @param array                         $form_values Values of named fields in the payment form.
	 * @param int                           $customer_id Stripe Customer ID.
	 * @return array
	 */
	function get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ) {
		$amount = isset( $form_values['simpay_amount'] )
			? $form_values['simpay_amount']
			: simpay_convert_amount_to_cents( $form->amount );

		$charge_args = array(
			'amount'      => $amount,
			'currency'    => $form->currency,
			'customer'    => $customer_id,
			'metadata'    => array_merge(
				array(
					'simpay_form_id' => $form->id,
				),
				Legacy\Hooks\simpay_payment_metadata( $form, $form_data, $form_values, $customer_id )
			),
			'description' => Legacy\Hooks\simpay_payment_description( $form, $form_data, $form_values, $customer_id ),
		);

		if ( ! empty( $form->statement_descriptor ) ) {
			$charge_args['statement_descriptor'] = $form->statement_descriptor;
		}

		/**
		 * Filters arguments used to create a Charge from a Payment Form request.
		 *
		 * @since 3.9.0
		 *
		 * @param array                         $session_args Checkout Session args.
		 * @param SimplePay\Core\Abstracts\Form $form Form instance.
		 * @param array                         $form_data Form data generated by the client.
		 * @param array                         $form_values Values of named fields in the payment form.
		 * @param int                           $customer_id Stripe Customer ID.
		 */
		return apply_filters(
			'simpay_get_charge_args_from_payment_form_request',
			$charge_args,
			$form,
			$form_data,
			$form_values,
			$customer_id
		);
	}
}
