<?php
/**
 * REST API: Plaid Token Controller
 *
 * @package SimplePay\Core\REST_API\v2
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.9.0
 */

namespace SimplePay\Pro\REST_API\v2;

use SimplePay\Core\REST_API\Controller;
use SimplePay\Pro\Payment_Methods;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Plaid_Token_Controller class.
 *
 * @since 3.9.0
 */
class Plaid_Token_Controller extends Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpsp/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'plaid-token';

	/**
	 * Register the routes for Checkout Session.
	 *
	 * @since 3.9.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			$this->rest_base . '/create-link-token',
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_link_token' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);
	}

	/**
	 * Allows requests originating from a payment form.
	 *
	 * @since 3.9.0
	 *
	 * @param \WP_REST_Request Request data.
	 * @return bool
	 */
	public function create_item_permissions_check( $request ) {
		$form_values = $request['form_values'];

		if ( ! isset( $form_values['_wpnonce'] ) || ! wp_verify_nonce( $form_values['_wpnonce'], 'simpay_payment_form' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Handle an incoming request to create a Plaid Link token.
	 *
	 * @since 3.9.3
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type string $form_id Payment Form ID.
	 * }
	 * @return \WP_REST_Response
	 */
	public function create_link_token( $request ) {
		try {
			// Locate form.
			if ( ! isset( $request['form_id'] ) ) {
				throw new \Exception( __( 'Unable to locate payment form.', 'simple-pay' ) );
			}

			// Get form instance.
			$form_id = $request['form_id'];

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			$ach_settings = Payment_Methods\get_form_payment_method_settings( $form, 'ach-debit' );

			if ( empty( $ach_settings ) ) {
				throw new \Exception( __( 'ACH Debit is improperly configured not enabled on this form.', 'simple-pay' ) );
			}

			// Random user ID.
			$user = wp_generate_password();

			$plaid_mode = false === $form->is_livemode()
				? 'sandbox'
				: 'production';

			$client_name = ! empty( $form->company_name )
				? $form->company_name
				: get_bloginfo( 'name' );

			// Retrieve an Account ID.
			$args = array(
				'user'          => array(
					'client_user_id' => $user,
				),
				'client_name'   => $client_name,
				'products'      => array(
					'auth',
				),
				'country_codes' => array(
					'US',
				),
				'language'     => 'en',
				'client_id'    => $ach_settings['client_id'],
				'secret'       => $ach_settings['secret_key'][ $plaid_mode ],
			);

			$link_token_request = wp_remote_post(
				'https://' . $plaid_mode . '.plaid.com/link/token/create',
				array_merge(
					$this->get_request_args(),
					array(
						'body' => wp_json_encode( $args ),
					)
				)
			);

			if ( is_wp_error( $link_token_request ) ) {
				throw new \Exception( $link_token_request->get_error_message() );
			}

			$link_token_request = json_decode( wp_remote_retrieve_body( $link_token_request ), true );

			if ( isset( $link_token_request['error_code'] ) ) {
				throw new \Exception( $link_token_request['error_message'] );
			}

			return $link_token_request;
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => $e->getMessage(),
				),
				400
			);
		}
	}

	/**
	 * Handle an incoming request to create a Plaid Link access token.
	 *
	 * @since 3.9.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type string $form_id Payment Form ID.
	 *   @type string $public_token Plaid Link public token (generated).
	 *   @type string $account_id Selected account ID.
	 * }
	 * @return \WP_REST_Response
	 */
	public function create_item( $request ) {
		try {
			// Public Token.
			if ( ! isset( $request['public_token'] ) ) {
				throw new \Exception( __( 'Unable to locate payment token.', 'simple-pay' ) );
			}

			// Account ID.
			if ( ! isset( $request['account_id'] ) ) {
				throw new \Exception( __( 'Unable to locate account ID.', 'simple-pay' ) );
			}

			// Locate form.
			if ( ! isset( $request['form_id'] ) ) {
				throw new \Exception( __( 'Unable to locate payment form.', 'simple-pay' ) );
			}

			// Get form instance.
			$form_id = $request['form_id'];

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			$ach_settings = Payment_Methods\get_form_payment_method_settings( $form, 'ach-debit' );

			if ( empty( $ach_settings ) ) {
				throw new \Exception( __( 'ACH Debit is improperly configured not enabled on this form.', 'simple-pay' ) );
			}

			$plaid_mode = false === $form->is_livemode()
				? 'sandbox'
				: 'production';

			// Retrieve an Account ID.
			$args = array(
				'client_id'    => $ach_settings['client_id'],
				'secret'       => $ach_settings['secret_key'][ $plaid_mode ],
				'public_token' => $request['public_token'],
			);

			$access_token_request = wp_remote_post(
				'https://' . $plaid_mode . '.plaid.com/item/public_token/exchange',
				array_merge(
					$this->get_request_args(),
					array(
						'body' => wp_json_encode( $args ),
					)
				)
			);

			if ( is_wp_error( $access_token_request ) ) {
				throw new \Exception( $access_token_request->get_error_message() );
			}

			$access_token_response = json_decode( wp_remote_retrieve_body( $access_token_request ), true );

			if ( isset( $access_token_response['error_code'] ) ) {
				throw new \Exception( $access_token_response['error_message'] );
			}

			// Create a Bank Account Token.
			$args = array(
				'client_id'    => $ach_settings['client_id'],
				'secret'       => $ach_settings['secret_key'][ $plaid_mode ],
				'access_token' => $access_token_response['access_token'],
				'account_id'   => $request['account_id'],
			);

			$bank_token_request = wp_remote_post(
				'https://' . $plaid_mode . '.plaid.com/processor/stripe/bank_account_token/create',
				array_merge(
					$this->get_request_args(),
					array(
						'body' => wp_json_encode( $args ),
					)
				)
			);

			if ( is_wp_error( $bank_token_request ) ) {
				throw new \Exception( $bank_token_request->get_error_message() );
			}

			$bank_token_response = json_decode( wp_remote_retrieve_body( $bank_token_request ), true );

			if ( isset( $bank_token_response['error_code'] ) ) {
				throw new \Exception( $bank_token_response['error_message'] );
			}

			// Return response, including Bank Account Token.
			return $bank_token_response;
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => $e->getMessage(),
				),
				400
			);
		}
	}

	/**
	 * Returns arguments used for wp_remote_post requests.
	 *
	 * @since 3.9.3
	 *
	 * @return array
	 */
	private function get_request_args() {
		return array(
			'headers'   => array(
				'Content-Type' => 'application/json; charset=utf-8',
			),
			'sslverify' => false,
			'method'    => 'POST',
		);
	}
}
