<?php
/**
 * REST API: Subscription Controller
 *
 * @package SimplePay\Core\REST_API\v2
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.6.0
 */

namespace SimplePay\Pro\REST_API\v2;

use SimplePay\Core\Forms\Default_Form;
use SimplePay\Core\REST_API\Controller;
use SimplePay\Pro\Payments as Pro_Payments;
use SimplePay\Core\Payments as Core_Payments;
use SimplePay\Core\Legacy;
use SimplePay\Core\Utils;
use SimplePay\Pro\Legacy as Pro_Legacy;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Subscription_Controller class.
 *
 * @since 3.6.0
 */
class Subscription_Controller extends Controller {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'wpsp/v2';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'subscription';

	/**
	 * Register the routes for Checkout Session.
	 *
	 * @since 3.6.0
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			$this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			$this->rest_base . '/payment_method/(?P<subscription_id>[sub_[0-9a-z]+)/(?P<customer_id>[cus_[0-9a-z]+)',
			array(
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item_payment_method' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( false ),
				),
			)
		);
	}

	/**
	 * Allows requests originating from a payment form.
	 *
	 * @since 3.6.0
	 *
	 * @param \WP_REST_Request Request data.
	 * @return bool
	 */
	public function create_item_permissions_check( $request ) {
		$form_values = $request['form_values'];

		if ( ! isset( $form_values['_wpnonce'] ) || ! wp_verify_nonce( $form_values['_wpnonce'], 'simpay_payment_form' ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Handle an incoming request to create a Checkout Session.
	 *
	 * @since 3.6.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type int   $customer_id Customer ID previously generated with Payment Source.
	 *   @type int   $form_id Form ID used to generate PaymentIntent data.
	 *   @type array $form_data Client-generated formData information.
	 *   @type array $form_values Values of named fields in the payment form.
	 * }
	 * @return \WP_REST_Response
	 */
	public function create_item( $request ) {
		try {
			// Locate form.
			if ( ! isset( $request['form_id'] ) ) {
				throw new \Exception( __( 'Unable to locate payment form.', 'simple-pay' ) );
			}

			// Gather customer information.
			$customer_id = isset( $request['customer_id'] ) ? $request['customer_id'] : false;

			if ( ! $customer_id ) {
				throw new \Exception( __( 'A customer must be provided.', 'simple-pay' ) );
			}

			// Gather <form> information.
			$form_id     = $request['form_id'];
			$form_data   = $request['form_data'];
			$form_values = $request['form_values'];

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			// Handle legacy form processing.
			Legacy\Hooks\simpay_process_form( $form, $form_data, $form_values, $customer_id );

			$subscription_args = array_merge(
				Pro_Payments\Subscription\get_args_from_payment_form_request( $form, $form_data, $form_values, $customer_id ),
				array(
					'customer' => $customer_id,
					'expand'   => array(
						'latest_invoice.payment_intent',
						'customer',
						'pending_setup_intent',
					),
				)
			);

			// Add a setup fee if required.
			Pro_Payments\Subscription\add_setup_fee_from_payment_form_request( $form, $form_data, $form_values, $customer_id );

			/**
			 * Allow further processing before a Subscription is created from a posted form.
			 *
			 * @since 3.6.0
			 *
			 * @param array                         $subscription_args Arguments used to create a PaymentIntent.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int                           $customer_id Stripe Customer ID.
			 */
			do_action(
				'simpay_before_subscription_from_payment_form_request',
				$subscription_args,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			$subscription = Pro_Payments\Subscription\create(
				$subscription_args,
				$form->get_api_request_args()
			);

			/**
			 * Allow further processing after a Subscription is created from a posted form.
			 *
			 * @since 3.6.0
			 *
			 * @param \Stripe\Subscription          $subscription Stripe Subscription.
			 * @param SimplePay\Core\Abstracts\Form $form Form instance.
			 * @param array                         $form_data Form data generated by the client.
			 * @param array                         $form_values Values of named fields in the payment form.
			 * @param int                           $customer_id Stripe Customer ID.
			 */
			do_action(
				'simpay_after_subscription_from_payment_form_request',
				$subscription,
				$form,
				$form_data,
				$form_values,
				$customer_id
			);

			return $subscription;
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => Utils\handle_exception_message( $e ),
				),
				400
			);
		}
	}

	/**
	 * Allows requests originating from a payment method update form.
	 *
	 * @since 3.7.0
	 *
	 * @param \WP_REST_Request Request data.
	 * @return bool
	 */
	public function update_item_permissions_check( $request ) {
		return $this->create_item_permissions_check( $request );
	}

	/**
	 * Handles an incoming request to update a Subscription's payment method.
	 *
	 * @since 3.7.0
	 *
	 * @param \WP_REST_Request $request {
	 *   Incoming REQUEST data.
	 *
	 *   @type string $source_id Source ID.
	 *   @type string $customer_id Customer ID.
	 * }
	 * @return \WP_REST_Response
	 */
	public function update_item_payment_method( $request ) {
		$subscription_id = $request['subscription_id'];
		$customer_id     = $request['customer_id'];

		try {
			$form_id = isset( $request['form_id'] ) ? $request['form_id'] : null;

			if ( ! $form_id ) {
				throw new \Exception( __( 'Unable to find Payment Form.', 'simple-pay' ) );
			}

			/** This filter is documented in includes/core/shortcodes.php */
			$form = apply_filters( 'simpay_form_view', '', $form_id );

			if ( empty( $form ) ) {
				$form = new Default_Form( $form_id );
			}

			$form_values = $request['form_values'];

			$source_id        = isset( $form_values['source_id'] ) ? $form_values['source_id'] : null;
			$subscription_key = isset( $form_values['subscription_key'] ) ? $form_values['subscription_key'] : null;

			if ( ! $source_id ) {
				throw new \Exception( __( 'Unable to find source.', 'simple-pay' ) );
			}

			$subscription = Pro_Payments\Subscription\retrieve( $subscription_id );

			// Confirm the actual Subscription's Customer matches the request.
			if ( $subscription->customer !== $customer_id ) {
				throw new \Exception( __( 'Subscription Customer does not match.', 'simple-pay' ) );
			}

			// Confirm the passed Subscription Key  matches the actual Subscription record metadata.
			if ( ! isset( $subscription->metadata->simpay_subscription_key ) || $subscription_key !== $subscription->metadata->simpay_subscription_key ) {
				throw new \Exception( __( 'Invalid Subscription key.', 'simple-pay' ) );
			}

			// Update the base Customer payment source.
			$customer = Core_Payments\Customer\update(
				$customer_id,
				array(
					'source' => $source_id,
				),
				$form->get_api_request_args()
			);

			// Update the Subscription's invoice setting default payment method
			// only if using an actual Payment Method.
			if ( false !== strpos( $source_id, 'ba_' ) ) {
				$subscription = Pro_Payments\Subscription\update(
					$subscription_id,
					array(
						'default_payment_method' => $source_id,
					),
					$form->get_api_request_args()
				);
			}

			return array(
				'customer'     => $customer,
				'subscription' => $subscription,
			);
		} catch ( \Exception $e ) {
			return new \WP_REST_Response(
				array(
					'message' => Utils\handle_exception_message( $e ),
				),
				400
			);
		}
	}
}
