<?php
/**
 * Webhooks: Charge Failed
 *
 * @package SimplePay\Pro\Webhooks
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.9.0
 */

namespace SimplePay\Pro\Webhooks;

use SimplePay\Pro\Payments\Charge;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Webhook_Charge_Failed class.
 *
 * @since 3.9.0
 */
class Webhook_Charge_Failed extends Webhook_Base implements Webhook_Interface {

	/**
	 * Alerts when a Charge has failed.
	 *
	 * @since 3.9.0
	 */
	public function handle() {
		$charge = $this->event->data->object;

		// We can't safely proceed if we are unable to identify the Payment Form
		// this webhook originated from.
		if ( ! isset( $charge->metadata->simpay_form_id ) ) {
			return;
		}

		$form_id = $charge->metadata->simpay_form_id;

		/** This filter is documented in includes/core/shortcodes.php */
		$form = apply_filters( 'simpay_form_view', '', $form_id );

		if ( empty( $form ) ) {
			$form = new Default_Form( $form_id );
		}

		// Retreive again with Customer expanded.
		$charge = Charge\retrieve(
			array(
				'id'     => $charge->id,
				'expand' => array(
					'customer',
				),
			),
			$form->get_api_request_args()
		);

		/**
		 * Allows processing after a Charge fails.
		 *
		 * @since 3.9.0
		 *
		 * @param \Stripe\Event  $event  Stripe webhook event.
		 * @param \Stripe\Charge $charge Stripe Charge.
		 */
		do_action( 'simpay_webhook_charge_failed', $this->event, $this->charge );
	}
}
