<?php
/**
 * Webhooks: Checkout Session Completed
 *
 * @package SimplePay\Pro\Webhooks
 * @copyright Copyright (c) 2020, Sandhills Development, LLC
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since 3.7.0
 */

namespace SimplePay\Pro\Webhooks;

use SimplePay\Core\Payments\Customer;
use SimplePay\Core\Payments\PaymentIntent;
use SimplePay\Pro\Payments\Subscription;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Webhook_Checkout_Session_Completed class.
 *
 * @since 3.7.0
 */
class Webhook_Checkout_Session_Completed extends Webhook_Base implements Webhook_Interface {

	/**
	 * Customer.
	 *
	 * @since 3.7.0
	 * @type \Stripe\Customer
	 */
	public $customer = null;

	/**
	 * Payment Intent.
	 *
	 * @since 3.7.0
	 * @type \Stripe\PaymentIntent
	 */
	public $payment_intent = null;

	/**
	 * Subscription.
	 *
	 * @since 3.7.0
	 * @type \Stripe\Subscription
	 */
	public $subscription = null;

	/**
	 * Handle the Webhook's data.
	 *
	 * @since 3.7.0
	 */
	public function handle() {
		$object = $this->event->data->object;

		// We can't safely proceed if we are unable to identify the Payment Form
		// this webhook originated from.
		if ( ! isset( $object->metadata->simpay_form_id ) ) {
			return;
		}

		$form_id = $object->metadata->simpay_form_id;

		/** This filter is documented in includes/core/shortcodes.php */
		$form = apply_filters( 'simpay_form_view', '', $form_id );

		if ( empty( $form ) ) {
			$form = new Default_Form( $form_id );
		}

		if ( null !== $object->customer ) {
			$this->customer = Customer\retrieve(
				$object->customer,
				$form->get_api_request_args()
			);
		}

		if ( null !== $object->payment_intent ) {
			$this->payment_intent = PaymentIntent\retrieve(
				$object->payment_intent,
				$form->get_api_request_args()
			);
		}

		if ( null !== $object->subscription ) {
			$this->subscription = Subscription\retrieve(
				$object->subscription,
				$form->get_api_request_args()
			);
		}

		/**
		 * Allows processing after a Checkout Session is completed.
		 *
		 * @since 3.7.0
		 *
		 * @param \Stripe\Event              $event Stripe webhook event.
		 * @param null|\Stripe\Customer      $customer Stripe Customer.
		 * @param null|\Stripe\PaymentIntent $payment_intent Stripe PaymentIntent.
		 * @param null|\Stripe\Subscription  $subscription Stripe Subscription.
		 */
		do_action(
			'simpay_webhook_checkout_session_completed',
			$this->event,
			$this->customer,
			$this->payment_intent,
			$this->subscription
		);
	}
}
